<?php
/* Copyright (C) 2004-2014	Laurent Destailleur	<eldy@users.sourceforge.net>
 * Copyright (C) 2005-2012	Regis Houssin		<regis.houssin@inodbox.com>
 * Copyright (C) 2008		Raphael Bertrand		<raphael.bertrand@resultic.fr>
 * Copyright (C) 2010-2014	Juanjo Menent		<jmenent@2byte.es>
 * Copyright (C) 2012		Christophe Battarel	<christophe.battarel@altairis.fr>
 * Copyright (C) 2012		Cédric Salvador		<csalvador@gpcsolutions.fr>
 * Copyright (C) 2012-2014	Raphaël Doursenaud	<rdoursenaud@gpcsolutions.fr>
 * Copyright (C) 2015		Marcos García		<marcosgdf@gmail.com>
 * Copyright (C) 2017-2018	Ferran Marcet		<fmarcet@2byte.es>
 * Copyright (C) 2018-2020  Frédéric France     <frederic.france@netlogic.fr>
 * Copyright (C) 2022		Anthony Berton				<anthony.berton@bb2a.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 * or see https://www.gnu.org/
 */

/**
 *	\file       htdocs/core/modules/facture/doc/pdf_crabe.modules.php
 *	\ingroup    facture
 *	\brief      File of class to generate customers invoices from crabe model
 */

require_once DOL_DOCUMENT_ROOT . '/core/modules/facture/modules_facture.php';
require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
require_once DOL_DOCUMENT_ROOT . '/core/lib/company.lib.php';
require_once DOL_DOCUMENT_ROOT . '/core/lib/functions2.lib.php';
require_once DOL_DOCUMENT_ROOT . '/core/lib/pdf.lib.php';
dol_include_once('/cfdixml/lib/cfdixml.lib.php');
dol_include_once('/cfdixml/lib/phpqrcode/qrlib.php');
dol_include_once('/cfdixml/vendor/autoload.php');

class pdf_cfdixmlv2 extends ModelePDFFactures
{
	/**
	 * @var DoliDb Database handler
	 */
	public $db;

	/**
	 * @var string model name
	 */
	public $name;

	/**
	 * @var string model description (short text)
	 */
	public $description;

	/**
	 * @var int 	Save the name of generated file as the main doc when generating a doc with this template
	 */
	public $update_main_doc_field;

	/**
	 * @var string document type
	 */
	public $type;

	/**
	 * @var array Minimum version of PHP required by module.
	 * e.g.: PHP ≥ 7.0 = array(7, 0)
	 */
	public $phpmin = array(7, 0);

	/**
	 * Dolibarr version of the loaded document
	 * @var string
	 */
	public $version = 'dolibarr';

	/**
	 * @var int page_largeur
	 */
	public $page_largeur;

	/**
	 * @var int page_hauteur
	 */
	public $page_hauteur;

	/**
	 * @var array format
	 */
	public $format;

	/**
	 * @var int marge_gauche
	 */
	public $marge_gauche;

	/**
	 * @var int marge_droite
	 */
	public $marge_droite;

	/**
	 * @var int marge_haute
	 */
	public $marge_haute;

	/**
	 * @var int marge_basse
	 */
	public $marge_basse;

	/**
	 * Issuer
	 * @var Societe Object that emits
	 */
	public $emetteur;

	/**
	 * @var bool Situation invoice type
	 */
	public $situationinvoice;

	/**
	 * @var float X position for the situation progress column
	 */
	public $posxprogress;

	/**
	 * @var int Category of operation
	 */
	public $categoryOfOperation = -1; // unknown by default


	/**
	 *	Constructor
	 *
	 *  @param		DoliDB		$db      Database handler
	 */
	public function __construct($db)
	{
		global $conf, $langs, $mysoc;

		// Translations
		$langs->loadLangs(array("main", "bills", "cfdixml@cfdixml"));

		$this->db = $db;
		$this->name = "PDF CfdiXMLv2";
		$this->description = $langs->trans('PDFCfdiXMLv2');
		$this->update_main_doc_field = 1; // Save the name of generated file as the main doc when generating a doc with this template

		// Dimension page
		$this->type = 'pdf';
		$formatarray = pdf_getFormat();
		$this->page_largeur = $formatarray['width'];
		$this->page_hauteur = $formatarray['height'];
		$this->format = array($this->page_largeur, $this->page_hauteur);
		$this->marge_gauche = getDolGlobalInt('MAIN_PDF_MARGIN_LEFT', 10);
		$this->marge_droite = getDolGlobalInt('MAIN_PDF_MARGIN_RIGHT', 10);
		$this->marge_haute = getDolGlobalInt('MAIN_PDF_MARGIN_TOP', 10);
		$this->marge_basse = getDolGlobalInt('MAIN_PDF_MARGIN_BOTTOM', 10);

		$this->option_logo = 1; // Display logo
		$this->option_tva = 1; // Manage the vat option FACTURE_TVAOPTION
		$this->option_modereg = 1; // Display payment mode
		$this->option_condreg = 1; // Display payment terms
		$this->option_multilang = 1; // Available in several languages
		$this->option_escompte = 1; // Displays if there has been a discount
		$this->option_credit_note = 1; // Support credit notes
		$this->option_freetext = 1; // Support add of a personalised text
		$this->option_draft_watermark = 1; // Support add of a watermark on drafts
		$this->watermark = '';

		// Get source company
		$this->emetteur = $mysoc;
		if (empty($this->emetteur->country_code)) {
			$this->emetteur->country_code = substr($langs->defaultlang, -2); // By default, if was not defined
		}

		// Define position of columns
		$this->posxdesc = $this->marge_gauche + 1;
		if (!empty($conf->global->PRODUCT_USE_UNITS)) {
			$this->posxref = 1;
			$this->posxumed = 1;
			$this->posxclaveprodserv = 1;
			$this->posxtva = 101;
			$this->posxup = 118;
			$this->posxqty = 135;
			$this->posxunit = 151;
		} else {
			$this->posxref = 1;
			$this->posxumed = 1;
			$this->posxclaveprodserv = 1;
			$this->posxtva = 106;
			$this->posxup = 122;
			$this->posxqty = 145;
			$this->posxunit = 162;
		}
		$this->posxprogress = 151; // Only displayed for situation invoices
		$this->posxdiscount = 162;
		$this->posxprogress = 174;
		$this->postotalht = 174;
		if (!empty($conf->global->MAIN_GENERATE_DOCUMENTS_WITHOUT_VAT) || !empty($conf->global->MAIN_GENERATE_DOCUMENTS_WITHOUT_VAT_COLUMN)) {
			$this->posxtva = $this->posxup;
		}
		$this->posxpicture = $this->posxtva - (empty($conf->global->MAIN_DOCUMENTS_WITH_PICTURE_WIDTH) ? 20 : $conf->global->MAIN_DOCUMENTS_WITH_PICTURE_WIDTH); // width of images
		if ($this->page_largeur < 210) { // To work with US executive format
			$this->posxpicture -= 20;
			$this->posxtva -= 20;
			$this->posxup -= 20;
			$this->posxqty -= 20;
			$this->posxunit -= 20;
			$this->posxdiscount -= 20;
			$this->posxprogress -= 20;
			$this->postotalht -= 20;
		}

		$this->tva = array();
		$this->tva_array = array();
		$this->localtax1 = array();
		$this->localtax2 = array();
		$this->atleastoneratenotnull = 0;
		$this->atleastonediscount = 0;
		$this->situationinvoice = false;
	}


	/**
	 *  Function to build pdf onto disk
	 *
	 *  @param		Facture		$object				Object to generate
	 *  @param		Translate	$outputlangs		Lang output object
	 *  @param		string		$srctemplatepath	Full path of source filename for generator using a template file
	 *  @param		int			$hidedetails		Do not show line details
	 *  @param		int			$hidedesc			Do not show desc
	 *  @param		int			$hideref			Do not show ref
	 *  @return     int         	    			1=OK, 0=KO
	 */
	public function write_file($object, $outputlangs, $srctemplatepath = '', $hidedetails = 0, $hidedesc = 0, $hideref = 0)
	{
		global $user, $langs, $conf, $mysoc, $hookmanager, $nblines;

		dol_syslog("write_file outputlangs->defaultlang=" . (is_object($outputlangs) ? $outputlangs->defaultlang : 'null'));

		if (!is_object($outputlangs)) {
			$outputlangs = $langs;
		}
		// For backward compatibility with FPDF, force output charset to ISO, because FPDF expect text to be encoded in ISO
		if (!empty($conf->global->MAIN_USE_FPDF)) {
			$outputlangs->charset_output = 'ISO-8859-1';
		}

		// Load translation files required by the page
		$outputlangs->loadLangs(array("main", "bills", "products", "dict", "companies"));

		// Show Draft Watermark
		if ($object->statut == $object::STATUS_DRAFT && (!empty($conf->global->FACTURE_DRAFT_WATERMARK))) {
			$this->watermark = $conf->global->FACTURE_DRAFT_WATERMARK;
		}


		global $outputlangsbis;
		$outputlangsbis = null;
		if (!empty($conf->global->PDF_USE_ALSO_LANGUAGE_CODE) && $outputlangs->defaultlang != $conf->global->PDF_USE_ALSO_LANGUAGE_CODE) {
			$outputlangsbis = new Translate('', $conf);
			$outputlangsbis->setDefaultLang($conf->global->PDF_USE_ALSO_LANGUAGE_CODE);
			$outputlangsbis->loadLangs(array("main", "bills", "products", "dict", "companies"));
		}

		$nblines = count($object->lines);

		//Load XML

		$object->fetch_optionals();
		$xml = $this->_cfdixmldata($object->array_options['options_cfdixml_xml']);
		if (empty($object->array_options['options_cfdixml_xml'])) {
			$this->watermark = $langs->transnoentities('CFDINOTVALID');;
		}

		if ($conf->facture->dir_output) {
			$object->fetch_thirdparty();

			$deja_regle = $object->getSommePaiement((isModEnabled("multicurrency") && $object->multicurrency_tx != 1) ? 1 : 0);
			$amount_credit_notes_included = $object->getSumCreditNotesUsed((isModEnabled("multicurrency") && $object->multicurrency_tx != 1) ? 1 : 0);
			$amount_deposits_included = $object->getSumDepositsUsed((isModEnabled("multicurrency") && $object->multicurrency_tx != 1) ? 1 : 0);

			// Definition of $dir and $file
			if ($object->specimen) {
				$dir = empty($conf->facture->multidir_output[$object->entity]) ? $conf->facture->dir_output : $conf->facture->multidir_output[$object->entity];
				$file = $dir . "/SPECIMEN.pdf";
			} else {
				$objectref = dol_sanitizeFileName($object->ref);
				$dir = empty($conf->facture->multidir_output[$object->entity]) ? $conf->facture->dir_output : $conf->facture->multidir_output[$object->entity];
				$dir .= "/" . $objectref;

				// Si tiene UUID, lo incluimos en el nombre del archivo
				if (!empty($object->array_options['options_cfdixml_UUID'])) {
					$file = $dir . "/" . $objectref . '_' . $object->array_options['options_cfdixml_UUID'] . ".pdf";
				} else {
					// Si no tiene UUID, usamos solo la referencia
					$file = $dir . "/" . $objectref . "_borrador.pdf";
				}
			}
			if (!file_exists($dir)) {
				if (dol_mkdir($dir) < 0) {
					$this->error = $langs->transnoentities("ErrorCanNotCreateDir", $dir);
					return 0;
				}
			}

			if (file_exists($dir)) {
				// Add pdfgeneration hook
				if (!is_object($hookmanager)) {
					include_once DOL_DOCUMENT_ROOT . '/core/class/hookmanager.class.php';
					$hookmanager = new HookManager($this->db);
				}
				$hookmanager->initHooks(array('pdfgeneration'));
				$parameters = array('file' => $file, 'object' => $object, 'outputlangs' => $outputlangs);
				global $action;
				$reshook = $hookmanager->executeHooks('beforePDFCreation', $parameters, $object, $action); // Note that $action and $object may have been modified by some hooks

				// Set nblines with the new facture lines content after hook
				$nblines = count($object->lines);
				$nbpayments = count($object->getListOfPayments());

				// Create pdf instance
				$pdf = pdf_getInstance($this->format);
				$default_font_size = pdf_getPDFFontSize($outputlangs); // Must be after pdf_getInstance
				$pdf->SetAutoPageBreak(1, 0);

				$heightforinfotot = 50 + (4 * $nbpayments); // Height reserved to output the info and total part and payment part
				if ($heightforinfotot > 220) {
					$heightforinfotot = 220;
				}
				$heightforfreetext = (isset($conf->global->MAIN_PDF_FREETEXT_HEIGHT) ? $conf->global->MAIN_PDF_FREETEXT_HEIGHT : 5); // Height reserved to output the free text on last page
				$heightforfooter = $this->marge_basse + 8; // Height reserved to output the footer (value include bottom margin)
				if (!empty($conf->global->MAIN_GENERATE_DOCUMENTS_SHOW_FOOT_DETAILS)) {
					$heightforfooter += 6;
				}

				if (class_exists('TCPDF')) {
					$pdf->setPrintHeader(false);
					$pdf->setPrintFooter(false);
				}
				$pdf->SetFont(pdf_getPDFFont($outputlangs));

				// Set path to the background PDF File
				if (!empty($conf->global->MAIN_ADD_PDF_BACKGROUND)) {
					$logodir = $conf->mycompany->dir_output;
					if (!empty($conf->mycompany->multidir_output[$object->entity])) {
						$logodir = $conf->mycompany->multidir_output[$object->entity];
					}
					$pagecount = $pdf->setSourceFile($logodir . '/' . $conf->global->MAIN_ADD_PDF_BACKGROUND);
					$tplidx = $pdf->importPage(1);
				}

				$pdf->Open();
				$pagenb = 0;
				$pdf->SetDrawColor(128, 128, 128);

				$pdf->SetTitle($outputlangs->convToOutputCharset($object->ref));
				$pdf->SetSubject($outputlangs->transnoentities("PdfInvoiceTitle"));
				$pdf->SetCreator("Dolibarr " . DOL_VERSION);
				$pdf->SetAuthor($mysoc->name . ($user->id > 0 ? ' - ' . $outputlangs->convToOutputCharset($user->getFullName($outputlangs)) : ''));
				$pdf->SetKeyWords($outputlangs->convToOutputCharset($object->ref) . " " . $outputlangs->transnoentities("PdfInvoiceTitle") . " " . $outputlangs->convToOutputCharset($object->thirdparty->name));
				if (getDolGlobalString('MAIN_DISABLE_PDF_COMPRESSION')) {
					$pdf->SetCompression(false);
				}

				// Set certificate
				$cert = empty($user->conf->CERTIFICATE_CRT) ? '' : $user->conf->CERTIFICATE_CRT;
				$certprivate = empty($user->conf->CERTIFICATE_CRT_PRIVATE) ? '' : $user->conf->CERTIFICATE_CRT_PRIVATE;
				// If user has no certificate, we try to take the company one
				if (!$cert) {
					$cert = empty($conf->global->CERTIFICATE_CRT) ? '' : $conf->global->CERTIFICATE_CRT;
				}
				if (!$certprivate) {
					$certprivate = empty($conf->global->CERTIFICATE_CRT_PRIVATE) ? '' : $conf->global->CERTIFICATE_CRT_PRIVATE;
				}
				// If a certificate is found
				if ($cert) {
					$info = array(
						'Name' => $this->emetteur->name,
						'Location' => getCountry($this->emetteur->country_code, 0),
						'Reason' => 'INVOICE',
						'ContactInfo' => $this->emetteur->email
					);
					$pdf->setSignature($cert, $certprivate, $this->emetteur->name, '', 2, $info);
				}

				$pdf->SetMargins($this->marge_gauche, $this->marge_haute, $this->marge_droite); // Left, Top, Right

				// Set $this->atleastonediscount if you have at least one discount
				// and determine category of operation
				$categoryOfOperation = 0;
				$nbProduct = 0;
				$nbService = 0;
				for ($i = 0; $i < $nblines; $i++) {
					if ($object->lines[$i]->remise_percent) {
						$this->atleastonediscount++;
					}

					// determine category of operation
					if ($categoryOfOperation < 2) {
						$lineProductType = $object->lines[$i]->product_type;
						if ($lineProductType == Product::TYPE_PRODUCT) {
							$nbProduct++;
						} elseif ($lineProductType == Product::TYPE_SERVICE) {
							$nbService++;
						}
						if ($nbProduct > 0 && $nbService > 0) {
							// mixed products and services
							$categoryOfOperation = 2;
						}
					}
				}
				// determine category of operation
				if ($categoryOfOperation <= 0) {
					// only services
					if ($nbProduct == 0 && $nbService > 0) {
						$categoryOfOperation = 1;
					}
				}
				$this->categoryOfOperation = $categoryOfOperation;
				if (empty($this->atleastonediscount)) {    // retrieve space not used by discount
					$delta = ($this->posxprogress - $this->posxdiscount);
					$this->posxpicture += $delta;
					$this->posxtva += $delta;
					$this->posxup += $delta;
					$this->posxqty += $delta;
					$this->posxunit += $delta;
					$this->posxdiscount += $delta;
					// post of fields after are not modified, stay at same position
				}

				$progress_width = 0;
				// Situation invoice handling
				if ($object->situation_cycle_ref && empty($conf->global->MAIN_PDF_HIDE_SITUATION)) {
					$this->situationinvoice = true;
					$progress_width = 10;
					$this->posxpicture -= $progress_width;
					$this->posxtva -= $progress_width;
					$this->posxup -= $progress_width;
					$this->posxqty -= $progress_width;
					$this->posxunit -= $progress_width;
					$this->posxdiscount -= $progress_width;
					$this->posxprogress -= $progress_width;
				}

				// New page
				$pdf->AddPage();
				if (!empty($tplidx)) {
					$pdf->useTemplate($tplidx);
				}
				$pagenb++;

				// Output header (logo, ref and address blocks). This is first call for first page.
				$top_shift = $this->_pagehead($pdf, $object, 1, $outputlangs, null, $xml);
				$pdf->SetFont('', '', $default_font_size - 1);
				$pdf->MultiCell(0, 3, ''); // Set interline to 3
				$pdf->SetTextColor(0, 0, 0);

				// $pdf->GetY() here can't be used. It is bottom of the second addresse box but first one may be higher

				// $tab_top is y where we must continue content (90 = 42 + 48: 42 is height of logo and ref, 48 is address blocks)
				$tab_top = 90 + $top_shift;		// top_shift is an addition for linked objects or addons (0 in most cases)
				$tab_top_newpage = (!getDolGlobalInt('MAIN_PDF_DONOTREPEAT_HEAD') ? 42 + $top_shift : 10);

				// You can add more thing under header here, if you increase $extra_under_address_shift too.
				$extra_under_address_shift = 0;
				$qrcodestring = '';
				if (!empty($conf->global->INVOICE_ADD_ZATCA_QR_CODE)) {
					$qrcodestring = $object->buildZATCAQRString();
				} elseif (!empty($conf->global->INVOICE_ADD_SWISS_QR_CODE)) {
					$qrcodestring = $object->buildSwitzerlandQRString();
				}
				if ($qrcodestring) {
					$qrcodecolor = array('25', '25', '25');
					// set style for QR-code
					$styleQr = array(
						'border' => false,
						'padding' => 0,
						'fgcolor' => $qrcodecolor,
						'bgcolor' => false, //array(255,255,255)
						'module_width' => 1, // width of a single module in points
						'module_height' => 1 // height of a single module in points
					);
					$pdf->write2DBarcode($qrcodestring, 'QRCODE,M', $this->marge_gauche, $tab_top - 5, 25, 25, $styleQr, 'N');
					$extra_under_address_shift += 25;
				}

				// Call hook printUnderHeaderPDFline
				$parameters = array(
					'object' => $object,
					'i' => $i,
					'pdf' => &$pdf,
					'outputlangs' => $outputlangs,
					'hidedetails' => $hidedetails
				);
				$reshook = $hookmanager->executeHooks('printUnderHeaderPDFline', $parameters, $this); // Note that $object may have been modified by hook
				if (!empty($hookmanager->resArray['extra_under_address_shift'])) {
					$extra_under_address_shift += $hookmanager->resArray['extra_under_header_shift'];
				}

				$tab_top += $extra_under_address_shift;
				$tab_top_newpage += 0;

				// Incoterm
				$height_incoterms = 0;

				//Factura CCE ???? no se ya veremos, pero este de momento no se toca
				if (isModEnabled('incoterm')) {
					$desc_incoterms = $object->getIncotermsForPDF();
					if ($desc_incoterms) {
						$tab_top -= 2;

						$pdf->SetFont('', '', $default_font_size - 1);
						$pdf->writeHTMLCell(190, 3, $this->posxdesc - 1, $tab_top - 1, dol_htmlentitiesbr($desc_incoterms), 0, 1);
						$nexY = $pdf->GetY();
						$height_incoterms = $nexY - $tab_top;

						// Rect takes a length in 3rd parameter
						$pdf->SetDrawColor(192, 192, 192);
						$pdf->Rect($this->marge_gauche, $tab_top - 1, $this->page_largeur - $this->marge_gauche - $this->marge_droite, $height_incoterms + 1);

						$tab_top = $nexY + 6;
					}
				}

				// Display notes
				$notetoshow = empty($object->note_public) ? '' : $object->note_public;
				if (!empty($conf->global->MAIN_ADD_SALE_REP_SIGNATURE_IN_NOTE)) {
					// Get first sale rep
					if (is_object($object->thirdparty)) {
						$salereparray = $object->thirdparty->getSalesRepresentatives($user);
						$salerepobj = new User($this->db);
						$salerepobj->fetch($salereparray[0]['id']);
						if (!empty($salerepobj->signature)) {
							$notetoshow = dol_concatdesc($notetoshow, $salerepobj->signature);
						}
					}
				}
				// Extrafields in note
				// $extranote = $this->getExtrafieldsInHtml($object, $outputlangs);
				// if (!empty($extranote)) {
				// 	$notetoshow = dol_concatdesc($notetoshow, $extranote);
				// }
				// if ($notetoshow) {
				// 	$tab_top -= 2;

				// 	$substitutionarray = pdf_getSubstitutionArray($outputlangs, null, $object);
				// 	complete_substitutions_array($substitutionarray, $outputlangs, $object);

				// 	$notetoshow = make_substitutions($notetoshow, $substitutionarray, $outputlangs);
				// 	$notetoshow = convertBackOfficeMediasLinksToPublicLinks($notetoshow);

				// 	$pdf->SetFont('', '', $default_font_size - 1);
				// 	$pdf->writeHTMLCell(190, 3, $this->posxdesc - 1, $tab_top - 1, dol_htmlentitiesbr($notetoshow), 0, 1);
				// 	$nexY = $pdf->GetY();
				// 	$height_note = $nexY - $tab_top;

				// 	// Rect takes a length in 3rd parameter
				// 	$pdf->SetDrawColor(192, 192, 192);
				// 	$pdf->Rect($this->marge_gauche, $tab_top - 1, $this->page_largeur - $this->marge_gauche - $this->marge_droite, $height_note + 1);

				// 	$tab_top = $nexY + 6;
				// }

				$iniY = $tab_top + 7;
				$curY = $tab_top + 7;
				$nexY = $tab_top + 7;

				// Loop on each lines
				for ($i = 0; $i < $nblines; $i++) {
					$curY = $nexY;
					$pdf->SetFont('', '', $default_font_size - 1); // Into loop to work with multipage
					$pdf->SetTextColor(0, 0, 0);

					// Define size of image if we need it
					$imglinesize = array();
					if (!empty($realpatharray[$i])) {
						$imglinesize = pdf_getSizeForImage($realpatharray[$i]);
					}

					$pdf->setTopMargin($tab_top_newpage);
					$pdf->setPageOrientation('', 1, $heightforfooter + $heightforfreetext + $heightforinfotot); // The only function to edit the bottom margin of current page to set it.
					$pageposbefore = $pdf->getPage();

					$showpricebeforepagebreak = 1;
					$posYAfterImage = 0;
					$posYAfterDescription = 0;

					// We start with Photo of product line
					if (isset($imglinesize['width']) && isset($imglinesize['height']) && ($curY + $imglinesize['height']) > ($this->page_hauteur - ($heightforfooter + $heightforfreetext + $heightforinfotot))) {	// If photo too high, we moved completely on new page
						$pdf->AddPage('', '', true);
						if (!empty($tplidx)) {
							$pdf->useTemplate($tplidx);
						}
						if (!getDolGlobalInt('MAIN_PDF_DONOTREPEAT_HEAD')) {
							$top_shift = $this->_pagehead($pdf, $object, 0, $outputlangs);
							$tab_top_newpage = (!getDolGlobalInt('MAIN_PDF_DONOTREPEAT_HEAD') ? 42 + $top_shift : 10);
						}
						$pdf->setPage($pageposbefore + 1);

						$curY = $tab_top_newpage;

						// Allows data in the first page if description is long enough to break in multiples pages
						if (!empty($conf->global->MAIN_PDF_DATA_ON_FIRST_PAGE)) {
							$showpricebeforepagebreak = 1;
						} else {
							$showpricebeforepagebreak = 0;
						}
					}

					if ($object->lines[$i]->fk_product > 0) {
						$ref = $object->lines[$i]->product_ref;
					} else {
						$ref = $langs->transnoentities('UnknownReference');
					}
					$curY = $curY + 2;
					$pdf->SetFont('', '', $default_font_size - 3);
					$pdf->SetXY($this->posxref + 5, $curY);
					$pdf->MultiCell($this->posxref + 18, 3, $ref, 0, 'C');
					$pdf->SetFont('', '', $default_font_size - 1);



					$umed = "";
					$clave = "";
					if ($object->lines[$i]->fk_product > 0) {
						$product = new Product($this->db);
						$product->fetch($object->lines[$i]->fk_product);
						$umed = $product->array_options['options_cfdixml_umed'];
						$clave = $product->array_options['options_cfdixml_claveprodserv'];
					} else {
						$umed = $object->lines[$i]->array_options['options_cfdixml_umed'];
						$clave = $object->lines[$i]->array_options['options_cfdixml_claveprodserv'];
					}
					$posx = $this->posxref + 25;
					//Clave

					$pdf->SetXY($posx, $curY);
					$pdf->MultiCell(18, 3, $clave, 0, 'C', 0);
					//Umed
					$posx += 18;
					$pdf->SetXY($posx, $curY);
					$pdf->MultiCell(15, 3, $umed, 0, 'C', 0);


					// Description of product line
					$pdf->startTransaction();

					$curX = $posx + 16.5;

					$textdata = pdf_writelinedesc($pdf, $object, $i, $outputlangs, 45, 3, $curX, $curY, 1, $conf->global->CFDIXML_HIDE_DESC);
					$anchoTexto = $pdf->GetStringWidth($textdata);

					$pageposafter = $pdf->getPage();
					if ($pageposafter > $pageposbefore) {	// There is a pagebreak
						$pdf->rollbackTransaction(true);
						$pageposafter = $pageposbefore;
						//print $pageposafter.'-'.$pageposbefore;exit;
						$pdf->setPageOrientation('', 1, $heightforfooter); // The only function to edit the bottom margin of current page to set it.
						pdf_writelinedesc($pdf, $object, $i, $outputlangs, 45, 3, $curX, $curY, 1, $conf->global->CFDIXML_HIDE_DESC);
						$anchoTexto = $pdf->GetStringWidth($textdata);
						$pageposafter = $pdf->getPage();
						$posyafter = $pdf->GetY();
						//var_dump($posyafter); var_dump(($this->page_hauteur - ($heightforfooter+$heightforfreetext+$heightforinfotot))); exit;
						if ($posyafter > ($this->page_hauteur - ($heightforfooter + $heightforfreetext + $heightforinfotot))) {	// There is no space left for total+free text
							if ($i == ($nblines - 1)) {	// No more lines, and no space left to show total, so we create a new page
								$pdf->AddPage('', '', true);
								if (!empty($tplidx)) {
									$pdf->useTemplate($tplidx);
								}
								if (!getDolGlobalInt('MAIN_PDF_DONOTREPEAT_HEAD')) {
									$top_shift = $this->_pagehead($pdf, $object, 0, $outputlangs);
									$tab_top_newpage = (!getDolGlobalInt('MAIN_PDF_DONOTREPEAT_HEAD') ? 42 + $top_shift : 10);
								}
								$pdf->setPage($pageposafter + 1);
							}
						} else {
							// We found a page break

							// Allows data in the first page if description is long enough to break in multiples pages
							if (!empty($conf->global->MAIN_PDF_DATA_ON_FIRST_PAGE)) {
								$showpricebeforepagebreak = 1;
							} else {
								$showpricebeforepagebreak = 0;
							}
						}
					} else // No pagebreak
					{
						$pdf->commitTransaction();
					}
					$posYAfterDescription = $pdf->GetY();

					$nexY = $pdf->GetY();
					$pageposafter = $pdf->getPage();
					$pdf->setPage($pageposbefore);
					$pdf->setTopMargin($this->marge_haute);
					$pdf->setPageOrientation('', 1, 0); // The only function to edit the bottom margin of current page to set it.

					// We suppose that a too long description or photo were moved completely on next page
					if ($pageposafter > $pageposbefore && empty($showpricebeforepagebreak)) {
						$pdf->setPage($pageposafter);
						$curY = $tab_top_newpage;
					}

					$pdf->SetFont('', '', $default_font_size - 1); // We reposition the default font
					$posx += $curX;

					// Quantity
					$qty = pdf_getlineqty($object, $i, $outputlangs, $hidedetails);
					$pdf->SetXY($posx, $curY);
					$pdf->MultiCell(13, 4, $qty, 0, 'C'); // Enough for 6 chars

					// // Discount on line
					$posx += 15;
					$remise_percent = 0;
					if ($object->lines[$i]->remise_percent) {
						$remise_percent = pdf_getlineremisepercent($object, $i, $outputlangs, $hidedetails);
					}
					$pdf->SetXY($posx - 2, $curY);
					$pdf->MultiCell(13, 3, $remise_percent, 0, 'C');

					// Unit price before discount
					$posx += 12.5;
					$up_excl_tax = pdf_getlineupexcltax($object, $i, $outputlangs, $hidedetails);
					$pdf->SetXY($posx, $curY);
					$pdf->MultiCell(22, 3, $up_excl_tax, 0, 'C', 0);

					$posx += 22;
					$pdf->SetXY($posx, $curY);
					$pdf->MultiCell(23.5, 3, number_format($object->lines[$i]->multicurrency_total_tva, 2, '.', ','), 0, 'C', 0);




					// Total HT line
					$total_excl_tax = pdf_getlinetotalexcltax($object, $i, $outputlangs, $hidedetails);
					$pdf->SetXY($this->postotalht + 4, $curY);
					$pdf->MultiCell(23.5, 3, $total_excl_tax, 0, 'C', 0);


					$sign = 1;
					if (isset($object->type) && $object->type == 2 && !empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE)) {
						$sign = -1;
					}
					// Collection of totals by value of VAT in $this->tva["taux"]=total_tva
					$prev_progress = $object->lines[$i]->get_prev_progress($object->id);
					if ($prev_progress > 0 && !empty($object->lines[$i]->situation_percent)) { // Compute progress from previous situation
						if (isModEnabled("multicurrency") && $object->multicurrency_tx != 1) {
							$tvaligne = $sign * $object->lines[$i]->multicurrency_total_tva * ($object->lines[$i]->situation_percent - $prev_progress) / $object->lines[$i]->situation_percent;
						} else {
							$tvaligne = $sign * $object->lines[$i]->total_tva * ($object->lines[$i]->situation_percent - $prev_progress) / $object->lines[$i]->situation_percent;
						}
					} else {
						if (isModEnabled("multicurrency") && $object->multicurrency_tx != 1) {
							$tvaligne = $sign * $object->lines[$i]->multicurrency_total_tva;
						} else {
							$tvaligne = $sign * $object->lines[$i]->total_tva;
						}
					}

					$localtax1ligne = $object->lines[$i]->total_localtax1;
					$localtax2ligne = $object->lines[$i]->total_localtax2;
					$localtax1_rate = $object->lines[$i]->localtax1_tx;
					$localtax2_rate = $object->lines[$i]->localtax2_tx;
					$localtax1_type = $object->lines[$i]->localtax1_type;
					$localtax2_type = $object->lines[$i]->localtax2_type;

					if ($object->remise_percent) {
						$tvaligne -= ($tvaligne * $object->remise_percent) / 100;
					}
					if ($object->remise_percent) {
						$localtax1ligne -= ($localtax1ligne * $object->remise_percent) / 100;
					}
					if ($object->remise_percent) {
						$localtax2ligne -= ($localtax2ligne * $object->remise_percent) / 100;
					}

					$vatrate = (string) $object->lines[$i]->tva_tx;

					// Retrieve type from database for backward compatibility with old records
					if ((!isset($localtax1_type) || $localtax1_type == '' || !isset($localtax2_type) || $localtax2_type == '') // if tax type not defined
						&& (!empty($localtax1_rate) || !empty($localtax2_rate))
					) { // and there is local tax
						$localtaxtmp_array = getLocalTaxesFromRate($vatrate, 0, $object->thirdparty, $mysoc);
						$localtax1_type = isset($localtaxtmp_array[0]) ? $localtaxtmp_array[0] : '';
						$localtax2_type = isset($localtaxtmp_array[2]) ? $localtaxtmp_array[2] : '';
					}

					// retrieve global local tax
					if ($localtax1_type && $localtax1ligne != 0) {
						if (empty($this->localtax1[$localtax1_type][$localtax1_rate])) {
							$this->localtax1[$localtax1_type][$localtax1_rate] = $localtax1ligne;
						} else {
							$this->localtax1[$localtax1_type][$localtax1_rate] += $localtax1ligne;
						}
					}
					if ($localtax2_type && $localtax2ligne != 0) {
						if (empty($this->localtax2[$localtax2_type][$localtax2_rate])) {
							$this->localtax2[$localtax2_type][$localtax2_rate] = $localtax2ligne;
						} else {
							$this->localtax2[$localtax2_type][$localtax2_rate] += $localtax2ligne;
						}
					}

					if (($object->lines[$i]->info_bits & 0x01) == 0x01) {
						$vatrate .= '*';
					}

					// Fill $this->tva and $this->tva_array
					if (!isset($this->tva[$vatrate])) {
						$this->tva[$vatrate] = 0;
					}
					$this->tva[$vatrate] += $tvaligne;	// ->tva is abandonned, we use now ->tva_array that is more complete
					$vatcode = $object->lines[$i]->vat_src_code;
					if (empty($this->tva_array[$vatrate . ($vatcode ? ' (' . $vatcode . ')' : '')]['amount'])) {
						$this->tva_array[$vatrate . ($vatcode ? ' (' . $vatcode . ')' : '')]['amount'] = 0;
					}
					$this->tva_array[$vatrate . ($vatcode ? ' (' . $vatcode . ')' : '')] = array('vatrate' => $vatrate, 'vatcode' => $vatcode, 'amount' => $this->tva_array[$vatrate . ($vatcode ? ' (' . $vatcode . ')' : '')]['amount'] + $tvaligne);

					if ($posYAfterImage > $posYAfterDescription) {
						$nexY = $posYAfterImage;
					}

					// Add line
					if (!empty($conf->global->MAIN_PDF_DASH_BETWEEN_LINES) && $i < ($nblines - 1)) {
						$pdf->setPage($pageposafter);
						$pdf->SetLineStyle(array('dash' => '1,1', 'color' => array(80, 80, 80)));
						//$pdf->SetDrawColor(190,190,200);
						$pdf->line($this->marge_gauche, $nexY + 1, $this->page_largeur - $this->marge_droite, $nexY + 1);
						$pdf->SetLineStyle(array('dash' => 0));
					}

					$nexY += 2; // Add space between lines

					// Detect if some page were added automatically and output _tableau for past pages
					while ($pagenb < $pageposafter) {
						$pdf->setPage($pagenb);
						if ($pagenb == 1) {
							$this->_tableau($pdf, $tab_top, $this->page_hauteur - $tab_top - $heightforfooter, 0, $outputlangs, 0, 1, $object->multicurrency_code);
						} else {
							$this->_tableau($pdf, $tab_top_newpage, $this->page_hauteur - $tab_top_newpage - $heightforfooter, 0, $outputlangs, 1, 1, $object->multicurrency_code);
						}
						$this->_pagefoot($pdf, $object, $outputlangs, 1);
						$pagenb++;
						$pdf->setPage($pagenb);
						$pdf->setPageOrientation('', 1, 0); // The only function to edit the bottom margin of current page to set it.
						if (!getDolGlobalInt('MAIN_PDF_DONOTREPEAT_HEAD')) {
							$top_shift = $this->_pagehead($pdf, $object, 0, $outputlangs);
							$tab_top_newpage = (!getDolGlobalInt('MAIN_PDF_DONOTREPEAT_HEAD') ? 42 + $top_shift : 10);
						}
						if (!empty($tplidx)) {
							$pdf->useTemplate($tplidx);
						}
					}
					if (isset($object->lines[$i + 1]->pagebreak) && $object->lines[$i + 1]->pagebreak) {
						if ($pagenb == 1) {
							$this->_tableau($pdf, $tab_top, $this->page_hauteur - $tab_top - $heightforfooter, 0, $outputlangs, 0, 1, $object->multicurrency_code);
						} else {
							$this->_tableau($pdf, $tab_top_newpage, $this->page_hauteur - $tab_top_newpage - $heightforfooter, 0, $outputlangs, 1, 1, $object->multicurrency_code);
						}
						$this->_pagefoot($pdf, $object, $outputlangs, 1);
						// New page
						$pdf->AddPage();
						if (!empty($tplidx)) {
							$pdf->useTemplate($tplidx);
						}
						$pagenb++;
						if (!getDolGlobalInt('MAIN_PDF_DONOTREPEAT_HEAD')) {
							$top_shift = $this->_pagehead($pdf, $object, 0, $outputlangs);
							$tab_top_newpage = (!getDolGlobalInt('MAIN_PDF_DONOTREPEAT_HEAD') ? 42 + $top_shift : 10);
						}
					}
				}

				// Show square
				if ($pagenb == 1) {
					$this->_tableau($pdf, $tab_top, $this->page_hauteur - $tab_top - $heightforinfotot - $heightforfreetext - $heightforfooter, 0, $outputlangs, 0, 0, $object->multicurrency_code);
					$bottomlasttab = $this->page_hauteur - $heightforinfotot - $heightforfreetext - $heightforfooter + 1;
				} else {
					$this->_tableau($pdf, $tab_top_newpage, $this->page_hauteur - $tab_top_newpage - $heightforinfotot - $heightforfreetext - $heightforfooter, 0, $outputlangs, 1, 0, $object->multicurrency_code);
					$bottomlasttab = $this->page_hauteur - $heightforinfotot - $heightforfreetext - $heightforfooter + 1;
				}
				dol_syslog("bottomlasttab=" . $bottomlasttab . " this->page_hauteur=" . $this->page_hauteur . " heightforinfotot=" . $heightforinfotot . " heightforfreetext=" . $heightforfreetext . " heightforfooter=" . $heightforfooter);

				// Display info area
				$posy = $this->_tableau_info($pdf, $object, $bottomlasttab, $outputlangs, $outputlangsbis, $xml);

				// Display total area
				$posy = $this->_tableau_tot($pdf, $object, $deja_regle, $bottomlasttab, $outputlangs, $outputlangsbis);

				// Display Payments area
				if (($deja_regle || $amount_credit_notes_included || $amount_deposits_included) && empty($conf->global->INVOICE_NO_PAYMENT_DETAILS)) {
					$posy = $this->_tableau_versements($pdf, $object, $posy, $outputlangs, $heightforfooter);
				}

				// Pagefoot
				$this->_pagefoot($pdf, $object, $outputlangs);
				if (method_exists($pdf, 'AliasNbPages')) {
					$pdf->AliasNbPages();
				}

				$pdf->Close();

				$pdf->Output($file, 'F');

				// Add pdfgeneration hook
				$hookmanager->initHooks(array('pdfgeneration'));
				$parameters = array('file' => $file, 'object' => $object, 'outputlangs' => $outputlangs);
				global $action;
				$reshook = $hookmanager->executeHooks('afterPDFCreation', $parameters, $this, $action); // Note that $action and $object may have been modified by some hooks
				if ($reshook < 0) {
					$this->error = $hookmanager->error;
					$this->errors = $hookmanager->errors;
				}

				if (!empty($conf->global->MAIN_UMASK)) {
					@chmod($file, octdec($conf->global->MAIN_UMASK));
				}

				$this->result = array('fullpath' => $file);

				return 1; // No error
			} else {
				$this->error = $langs->transnoentities("ErrorCanNotCreateDir", $dir);
				return 0;
			}
		} else {
			$this->error = $langs->transnoentities("ErrorConstantNotDefined", "FAC_OUTPUTDIR");
			return 0;
		}
	}

	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.PublicUnderscore
	/**
	 *  Show payments table
	 *
	 *  @param	TCPDF		$pdf            	Object PDF
	 *  @param  Facture		$object         	Object invoice
	 *  @param  int			$posy           	Position y in PDF
	 *  @param  Translate	$outputlangs    	Object langs for output
	 *  @param  int			$heightforfooter 	Height for footer
	 *  @return int             				<0 if KO, >0 if OK
	 */
	protected function _tableau_versements(&$pdf, $object, $posy, $outputlangs, $heightforfooter = 0) {}


	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.PublicUnderscore
	/**
	 * Function _tableau_versements_header
	 *
	 * @param TCPDF 		$pdf				Object PDF
	 * @param Facture		$object				Object invoice
	 * @param Translate		$outputlangs		Object langs for output
	 * @param int			$default_font_size	Font size
	 * @param int			$tab3_posx			pos x
	 * @param int 			$tab3_top			pos y
	 * @param int 			$tab3_width			width
	 * @param int 			$tab3_height		height
	 * @return void
	 */
	protected function _tableau_versements_header($pdf, $object, $outputlangs, $default_font_size, $tab3_posx, $tab3_top, $tab3_width, $tab3_height) {}

	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.PublicUnderscore
	/**
	 *   Show miscellaneous information (payment mode, payment term, ...)
	 *
	 *   @param		TCPDF		$pdf     		Object PDF
	 *   @param		Facture		$object			Object to show
	 *   @param		int			$posy			Y
	 *   @param		Translate	$outputlangs	Langs object
	 *   @param  	Translate	$outputlangsbis	Object lang for output bis
	 *   @return	int							Pos y
	 */
	protected function _tableau_info(&$pdf, $object, $posy, $outputlangs, $outputlangsbis, $xml)
	{
		global $conf;

		$default_font_size = pdf_getPDFFontSize($outputlangs);

		//UUID
		$pdf->SetFont('', 'B', $default_font_size - 2);
		$pdf->SetXY($this->marge_gauche, $posy);
		$pdf->MultiCell(10, 4, 'UUID: ', 0, 'L');

		$pdf->SetFont('', '', $default_font_size - 2);
		$posx = $pdf->GetX() + 10;
		$pdf->SetXY($posx, $posy);
		$pdf->MultiCell(70, 4, $object->array_options['options_cfdixml_UUID'], 0, 'L');

		//Cert SAT
		$pdf->SetFont('', 'B', $default_font_size - 2);
		$posx = $pdf->GetX() + 80;
		$pdf->SetXY($posx, $posy);
		$pdf->MultiCell(50, 4, 'Cert SAT: ', 0, 'L');

		$pdf->SetFont('', '', $default_font_size - 2);
		$posx = $pdf->GetX() + 95;
		$pdf->SetXY($posx, $posy);
		$pdf->MultiCell(80, 4, $object->array_options['options_cfdixml_certsat'], 0, 'L');


		//Cer Emisor
		$pdf->SetFont('', 'B', $default_font_size - 2);
		$pdf->SetXY($posx + 40, $posy);
		$pdf->MultiCell(60, 4, 'Cert Emisor: ', 0, 'L');

		$pdf->SetFont('', '', $default_font_size - 2);
		$pdf->SetXY($posx + 60, $posy);
		$pdf->MultiCell(40, 4, $xml['NoCertificado'], 0, 'L');

		$posx = $pdf->GetX();
		$posy = $pdf->Gety();
		//Fecha Emision
		$pdf->SetFont('', 'B', $default_font_size - 2);
		$pdf->SetXY($posx, $posy);
		$pdf->MultiCell(20, 4, 'F. Emisión: ', 0, 'L');
		//Cambiar fecha emisión
		$pdf->SetFont('', '', $default_font_size - 2);
		$pdf->SetXY($posx + 18, $posy);
		$pdf->MultiCell(35, 4, $object->array_options['options_cfdixml_fechatimbrado'], 0, 'L');
		//Fecha Certificación
		$pdf->SetFont('', 'B', $default_font_size - 2);
		$pdf->SetXY($posx + 50, $posy);
		$pdf->MultiCell(25, 4, 'F. Certificación: ', 0, 'L');

		$pdf->SetFont('', '', $default_font_size - 2);
		$pdf->SetXY($posx + 75, $posy);
		$pdf->MultiCell(35, 4, $object->array_options['options_cfdixml_fechatimbrado'], 0, 'L');


		//Sello Digital Emisor
		$posy = $pdf->Gety();
		$pdf->SetFont('', 'B', $default_font_size - 2);
		$pdf->SetXY($this->marge_gauche, $posy);
		$pdf->MultiCell(80, 4, 'Sello digital del emisor: ', 0, 'L');

		$posy = $pdf->Gety();
		$pdf->SetFont('', '', $default_font_size - 3);
		$pdf->SetXY(5, $posy);
		$pdf->MultiCell(150, 4, $object->array_options['options_cfdixml_sellocfd'], 0, 'J');

		//Sello Digital SAT
		$posy = $pdf->Gety();
		$pdf->SetFont('', 'B', $default_font_size - 2);
		$pdf->SetXY($this->marge_gauche, $posy);
		$pdf->MultiCell(80, 4, 'Sello digital del SAT: ', 0, 'L');

		$posy = $pdf->Gety();
		$pdf->SetFont('', '', $default_font_size - 3);
		$pdf->SetXY(5, $posy);
		$pdf->MultiCell(150, 4, $object->array_options['options_cfdixml_sellosat'], 0, 'J');

		//Sello Digital SAT
		$posy = $pdf->Gety();
		$pdf->SetFont('', 'B', $default_font_size - 2);
		$pdf->SetXY($this->marge_gauche, $posy);
		$pdf->MultiCell(80, 4, 'Cadena original: ', 0, 'L');

		$posy = $pdf->Gety();
		$pdf->SetFont('', '', $default_font_size - 3);
		$pdf->SetXY(5, $posy);
		$pdf->MultiCell(150, 4, $object->array_options['options_cfdixml_cadenaorig'], 0, 'J');
		$posy = $pdf->Gety();

		return $posy;
	}

	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.PublicUnderscore
	/**
	 *	Show total to pay
	 *
	 *	@param	TCPDF		$pdf            Object PDF
	 *	@param  Facture		$object         Object invoice
	 *	@param  int			$deja_regle     Amount already paid (in the currency of invoice)
	 *	@param	int			$posy			Position depart
	 *	@param	Translate	$outputlangs	Objet langs
	 *  @param  Translate	$outputlangsbis	Object lang for output bis
	 *	@return int							Position pour suite
	 */
	protected function _tableau_tot(&$pdf, $object, $deja_regle, $posy, $outputlangs, $outputlangsbis)
	{
		// phpcs:enable
		global $conf, $mysoc, $hookmanager, $emisor, $receptor;

		$sign = 1;
		if ($object->type == 2 && !empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE)) {
			$sign = -1;
		}
		$pdf->SetTextColor(0, 0, 0);
		$default_font_size = pdf_getPDFFontSize($outputlangs);

		$outputlangsbis = null;
		if (!empty($conf->global->PDF_USE_ALSO_LANGUAGE_CODE) && $outputlangs->defaultlang != $conf->global->PDF_USE_ALSO_LANGUAGE_CODE) {
			$outputlangsbis = new Translate('', $conf);
			$outputlangsbis->setDefaultLang($conf->global->PDF_USE_ALSO_LANGUAGE_CODE);
			$outputlangsbis->loadLangs(array("main", "dict", "companies", "bills", "products", "propal"));
			$default_font_size--;
		}

		$tab2_top = $posy - 15;
		$tab2_hl = 4;
		$pdf->SetFont('', '', $default_font_size - 1);

		// Total table
		$col1x = 120;
		$col2x = 170;
		if ($this->page_largeur < 210) { // To work with US executive format
			$col1x -= 15;
			$col2x -= 10;
		}
		$largcol2 = ($this->page_largeur - $this->marge_droite - $col2x);

		$useborder = 0;
		$index = 0;

		// Total HT
		// $pdf->SetFillColor(255, 255, 255);
		$pdf->SetXY($col1x, $tab2_top + 0);
		$pdf->MultiCell($col2x - $col1x, $tab2_hl, $outputlangs->transnoentities(empty($conf->global->MAIN_GENERATE_DOCUMENTS_WITHOUT_VAT) ? "TotalHT" : "Total") . (is_object($outputlangsbis) ? ' / ' . $outputlangsbis->transnoentities(empty($conf->global->MAIN_GENERATE_DOCUMENTS_WITHOUT_VAT) ? "TotalHT" : "Total") : ''), 0, 'L', 1);

		$total_ht = ((isModEnabled("multicurrency") && isset($object->multicurrency_tx) && $object->multicurrency_tx != 1) ? $object->multicurrency_total_ht : $object->total_ht);
		$pdf->SetXY($col2x, $tab2_top + 0);
		$pdf->MultiCell($largcol2, $tab2_hl, price($sign * ($total_ht + (!empty($object->remise) ? $object->remise : 0)), 0, $outputlangs), 0, 'R', 1);

		// Show VAT by rates and total
		// $pdf->SetFillColor(248, 248, 248);

		$total_ttc = (isModEnabled("multicurrency") && $object->multicurrency_tx != 1) ? $object->multicurrency_total_ttc : $object->total_ttc;

		$this->atleastoneratenotnull = 0;

		$tvaisnull = ((!empty($this->tva) && count($this->tva) == 1 && isset($this->tva['0.000']) && is_float($this->tva['0.000'])) ? true : false);
		if (!empty($conf->global->MAIN_GENERATE_DOCUMENTS_WITHOUT_VAT_IFNULL) && $tvaisnull) {
			// Nothing to do
		} else {
			// FIXME amount of vat not supported with multicurrency

			//Local tax 1 before VAT
			//if (!empty($conf->global->FACTURE_LOCAL_TAX1_OPTION) && $conf->global->FACTURE_LOCAL_TAX1_OPTION=='localtax1on')
			//{
			foreach ($this->localtax1 as $localtax_type => $localtax_rate) {
				if (in_array((string) $localtax_type, array('1', '3', '5'))) {
					continue;
				}

				foreach ($localtax_rate as $tvakey => $tvaval) {
					if ($tvakey != 0) {    // On affiche pas taux 0
						//$this->atleastoneratenotnull++;

						$index++;
						$pdf->SetXY($col1x, $tab2_top + $tab2_hl * $index);

						$tvacompl = '';
						if (preg_match('/\*/', $tvakey)) {
							$tvakey = str_replace('*', '', $tvakey);
							$tvacompl = " (" . $outputlangs->transnoentities("NonPercuRecuperable") . ")";
						}

						$totalvat = $outputlangs->transcountrynoentities("TotalLT1", $mysoc->country_code) . (is_object($outputlangsbis) ? ' / ' . $outputlangsbis->transcountrynoentities("TotalLT1", $mysoc->country_code) : '');
						$totalvat .= ' ';
						$totalvat .= vatrate(abs($tvakey), 1) . $tvacompl;
						$pdf->MultiCell($col2x - $col1x, $tab2_hl, $totalvat, 0, 'L', 1);

						$pdf->SetXY($col2x, $tab2_top + $tab2_hl * $index);
						$pdf->MultiCell($largcol2, $tab2_hl, price($tvaval, 0, $outputlangs), 0, 'R', 1);
					}
				}
			}
			//}
			//Local tax 2 before VAT
			//if (!empty($conf->global->FACTURE_LOCAL_TAX2_OPTION) && $conf->global->FACTURE_LOCAL_TAX2_OPTION=='localtax2on')
			//{
			foreach ($this->localtax2 as $localtax_type => $localtax_rate) {
				if (in_array((string) $localtax_type, array('1', '3', '5'))) {
					continue;
				}

				foreach ($localtax_rate as $tvakey => $tvaval) {
					if ($tvakey != 0) {    // On affiche pas taux 0
						//$this->atleastoneratenotnull++;

						$index++;
						$pdf->SetXY($col1x, $tab2_top + $tab2_hl * $index);

						$tvacompl = '';
						if (preg_match('/\*/', $tvakey)) {
							$tvakey = str_replace('*', '', $tvakey);
							$tvacompl = " (" . $outputlangs->transnoentities("NonPercuRecuperable") . ")";
						}
						$totalvat = $outputlangs->transcountrynoentities("TotalLT2", $mysoc->country_code) . (is_object($outputlangsbis) ? ' / ' . $outputlangsbis->transcountrynoentities("TotalLT2", $mysoc->country_code) : '');
						$totalvat .= ' ';
						$totalvat .= vatrate(abs($tvakey), 1) . $tvacompl;
						$pdf->MultiCell($col2x - $col1x, $tab2_hl, $totalvat, 0, 'L', 1);

						$pdf->SetXY($col2x, $tab2_top + $tab2_hl * $index);
						$pdf->MultiCell($largcol2, $tab2_hl, price($tvaval, 0, $outputlangs), 0, 'R', 1);
					}
				}
			}

			//}

			// VAT
			foreach ($this->tva_array as $tvakey => $tvaval) {
				if ($tvakey != 0) {    // On affiche pas taux 0
					$this->atleastoneratenotnull++;

					$index++;
					$pdf->SetXY($col1x, $tab2_top + $tab2_hl * $index);

					$tvacompl = '';
					if (preg_match('/\*/', $tvakey)) {
						$tvakey = str_replace('*', '', $tvakey);
						$tvacompl = " (" . $outputlangs->transnoentities("NonPercuRecuperable") . ")";
					}
					$totalvat = $outputlangs->transcountrynoentities("TotalVAT", $mysoc->country_code) . (is_object($outputlangsbis) ? ' / ' . $outputlangsbis->transcountrynoentities("TotalVAT", $mysoc->country_code) : '');
					$totalvat .= ' ';
					if (getDolGlobalString('PDF_VAT_LABEL_IS_CODE_OR_RATE') == 'rateonly') {
						$totalvat .= vatrate($tvaval['vatrate'], 1) . $tvacompl;
					} elseif (getDolGlobalString('PDF_VAT_LABEL_IS_CODE_OR_RATE') == 'codeonly') {
						$totalvat .= $tvaval['vatcode'] . $tvacompl;
					} else {
						$totalvat .= vatrate($tvaval['vatrate'], 1) . ($tvaval['vatcode'] ? ' (' . $tvaval['vatcode'] . ')' : '') . $tvacompl;
					}
					$pdf->MultiCell($col2x - $col1x, $tab2_hl, $totalvat, 0, 'L', 1);

					$pdf->SetXY($col2x, $tab2_top + $tab2_hl * $index);
					$pdf->MultiCell($largcol2, $tab2_hl, price(price2num($tvaval['amount'], 'MT'), 0, $outputlangs), 0, 'R', 1);
				}
			}

			//Local tax 1 after VAT
			//if (!empty($conf->global->FACTURE_LOCAL_TAX1_OPTION) && $conf->global->FACTURE_LOCAL_TAX1_OPTION=='localtax1on')
			//{
			foreach ($this->localtax1 as $localtax_type => $localtax_rate) {
				if (in_array((string) $localtax_type, array('2', '4', '6'))) {
					continue;
				}

				foreach ($localtax_rate as $tvakey => $tvaval) {
					if ($tvakey != 0) {    // On affiche pas taux 0
						//$this->atleastoneratenotnull++;

						$index++;
						$pdf->SetXY($col1x, $tab2_top + $tab2_hl * $index);

						$tvacompl = '';
						if (preg_match('/\*/', $tvakey)) {
							$tvakey = str_replace('*', '', $tvakey);
							$tvacompl = " (" . $outputlangs->transnoentities("NonPercuRecuperable") . ")";
						}
						$totalvat = $outputlangs->transcountrynoentities("TotalLT1", $mysoc->country_code) . ' ';
						$totalvat .= vatrate(abs($tvakey), 1) . $tvacompl;

						$pdf->MultiCell($col2x - $col1x, $tab2_hl, $totalvat, 0, 'L', 1);
						$pdf->SetXY($col2x, $tab2_top + $tab2_hl * $index);
						$pdf->MultiCell($largcol2, $tab2_hl, price($tvaval, 0, $outputlangs), 0, 'R', 1);
					}
				}
			}
			//}
			//Local tax 2 after VAT
			//if (!empty($conf->global->FACTURE_LOCAL_TAX2_OPTION) && $conf->global->FACTURE_LOCAL_TAX2_OPTION=='localtax2on')
			//{
			foreach ($this->localtax2 as $localtax_type => $localtax_rate) {
				if (in_array((string) $localtax_type, array('2', '4', '6'))) {
					continue;
				}

				foreach ($localtax_rate as $tvakey => $tvaval) {
					//$this->atleastoneratenotnull++;

					$index++;
					$pdf->SetXY($col1x, $tab2_top + $tab2_hl * $index);

					$tvacompl = '';
					if (preg_match('/\*/', $tvakey)) {
						$tvakey = str_replace('*', '', $tvakey);
						$tvacompl = " (" . $outputlangs->transnoentities("NonPercuRecuperable") . ")";
					}
					$totalvat = $outputlangs->transcountrynoentities("TotalLT2", $mysoc->country_code) . ' ';

					$totalvat .= vatrate(abs($tvakey), 1) . $tvacompl;
					$pdf->MultiCell($col2x - $col1x, $tab2_hl, $totalvat, 0, 'L', 1);

					$pdf->SetXY($col2x, $tab2_top + $tab2_hl * $index);
					$pdf->MultiCell($largcol2, $tab2_hl, price($tvaval, 0, $outputlangs), 0, 'R', 1);
				}
			}
			//}

			// Revenue stamp
			if (price2num($object->revenuestamp) != 0) {
				$index++;
				$pdf->SetXY($col1x, $tab2_top + $tab2_hl * $index);
				$pdf->MultiCell($col2x - $col1x, $tab2_hl, $outputlangs->transnoentities("RevenueStamp"), $useborder, 'L', 1);

				$pdf->SetXY($col2x, $tab2_top + $tab2_hl * $index);
				$pdf->MultiCell($largcol2, $tab2_hl, price($sign * $object->revenuestamp), $useborder, 'R', 1);
			}

			// Total TTC
			$index++;
			$pdf->SetXY($col1x, $tab2_top + $tab2_hl * $index);
			$pdf->SetTextColor(0, 0, 0);
			// $pdf->SetFillColor(224, 224, 224);
			$pdf->MultiCell($col2x - $col1x, $tab2_hl, $outputlangs->transnoentities("TotalTTC"), $useborder, 'L', 1);

			$pdf->SetXY($col2x, $tab2_top + $tab2_hl * $index);
			$pdf->MultiCell($largcol2, $tab2_hl, price($sign * $total_ttc, 0, $outputlangs), $useborder, 'R', 1);

			// Retained warranty
			if ($object->displayRetainedWarranty()) {
				$pdf->SetTextColor(0, 0, 0);
				// $pdf->SetFillColor(255, 255, 255);

				$retainedWarranty = $object->getRetainedWarrantyAmount();
				$billedWithRetainedWarranty = $object->total_ttc - $retainedWarranty;

				// Billed - retained warranty
				$index++;
				$pdf->SetXY($col1x, $tab2_top + $tab2_hl * $index);
				$pdf->MultiCell($col2x - $col1x, $tab2_hl, $outputlangs->transnoentities("ToPayOn", dol_print_date($object->date_lim_reglement, 'day')), $useborder, 'L', 1);

				$pdf->SetXY($col2x, $tab2_top + $tab2_hl * $index);
				$pdf->MultiCell($largcol2, $tab2_hl, price($billedWithRetainedWarranty), $useborder, 'R', 1);

				// retained warranty
				$index++;
				$pdf->SetXY($col1x, $tab2_top + $tab2_hl * $index);

				$retainedWarrantyToPayOn = $outputlangs->transnoentities("RetainedWarranty") . ' (' . $object->retained_warranty . '%)';
				$retainedWarrantyToPayOn .= !empty($object->retained_warranty_date_limit) ? ' ' . $outputlangs->transnoentities("toPayOn", dol_print_date($object->retained_warranty_date_limit, 'day')) : '';

				$pdf->MultiCell($col2x - $col1x, $tab2_hl, $retainedWarrantyToPayOn, $useborder, 'L', 1);
				$pdf->SetXY($col2x, $tab2_top + $tab2_hl * $index);
				$pdf->MultiCell($largcol2, $tab2_hl, price($retainedWarranty), $useborder, 'R', 1);
			}
		}


		$pdf->SetTextColor(0, 0, 0);

		//NumToWord
		// Backguard Compatibility
		if (function_exists('isModEnabled')) {
			if (isModEnabled('numberwords')) {
				dol_include_once('/numberwords/core/substitutions/functions_numberwords.lib.php');

				$text = numberwords_getLabelFromNumber($outputlangs, $object->multicurrency_total_ttc, 1);
				$posx = $col2x;
				$posy = $tab2_top + $tab2_hl * $index;
				$pdf->SetXY(5, $posy);
				$pdf->MultiCell(110, $tab2_hl, $text, 0, 'L', 1);
			}
		}
		// Manejo seguro del QR code
		if (
			!empty($object->array_options["options_cfdixml_UUID"]) &&
			!empty($emisor["Rfc"]) &&
			!empty($object->thirdparty->idprof1) &&
			!empty($object->multicurrency_total_ttc) &&
			!empty($object->array_options["options_cfdixml_sellocfd"])
		) {

			$data_cbb = 'https://verificacfdi.facturaelectronica.sat.gob.mx/default.aspx?id=' .
				$object->array_options["options_cfdixml_UUID"] .
				'&re=' . $emisor["Rfc"] .
				'&rr=' . $object->thirdparty->idprof1 .
				'&tt=' . $object->multicurrency_total_ttc .
				'&fe=' . substr($object->array_options["options_cfdixml_sellocfd"], -8);

			QRcode::png($data_cbb, $conf->facture->dir_output . "/" . $object->ref . "/" . $object->ref . '_' .
				$object->array_options["options_cfdixml_UUID"] . ".png");

			$pdf->Image(
				$conf->facture->dir_output . "/" . $object->ref . "/" . $object->ref . '_' .
					$object->array_options["options_cfdixml_UUID"] . ".png",
				$col2x - 10,
				$tab2_top + $tab2_hl * $index + 16,
				50,
				50,
				'PNG',
				''
			);
		}
		$index++;
		return ($tab2_top + ($tab2_hl * $index));
	}

	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.PublicUnderscore
	/**
	 *   Show table for lines
	 *
	 *   @param		TCPDF		$pdf     		Object PDF
	 *   @param		string		$tab_top		Top position of table
	 *   @param		string		$tab_height		Height of table (rectangle)
	 *   @param		int			$nexY			Y (not used)
	 *   @param		Translate	$outputlangs	Langs object
	 *   @param		int			$hidetop		1=Hide top bar of array and title, 0=Hide nothing, -1=Hide only title
	 *   @param		int			$hidebottom		Hide bottom bar of array
	 *   @param		string		$currency		Currency code
	 *   @return	void
	 */
	protected function _tableau(&$pdf, $tab_top, $tab_height, $nexY, $outputlangs, $hidetop = 0, $hidebottom = 0, $currency = '')
	{
		global $conf;

		// Force to disable hidetop and hidebottom
		$hidebottom = 0;
		if ($hidetop) {
			$hidetop = -1;
		}

		$currency = !empty($currency) ? $currency : $conf->currency;
		$default_font_size = pdf_getPDFFontSize($outputlangs);
		$pdf->SetFont('', '', $default_font_size - 1);

		$totalAncho = $pdf->GetPageWidth() - PDF_MARGIN_LEFT - PDF_MARGIN_RIGHT;
		$anchoPequeno = $totalAncho * 0.1; // 10% del ancho disponible
		$anchoDescripcion = $totalAncho * 0.25; // 40% del ancho disponible
		$anchoRestante = ($totalAncho - $anchoDescripcion - 5 * $anchoPequeno) / 2; // Distribuir el resto

		$pdf->SetFillColor(255, 255, 255);
		$pdf->SetTextColor(0, 0, 0);
		$pdf->SetXY($this->posxref + 5, $tab_top + 1);
		$pdf->Cell($anchoPequeno, 7, 'Ref.', 1, 0, 'C', 1);

		// Imprimir los encabezados
		$pdf->Cell($anchoPequeno, 7, 'Clave', 1, 0, 'C', 1);
		$pdf->Cell($anchoPequeno, 7, 'Unidad', 1, 0, 'C', 1);
		$pdf->Cell($anchoDescripcion, 7, 'Descripción', 1, 0, 'C', 1);

		$pdf->Cell($anchoPequeno - 5, 7, 'Cant.', 1, 0, 'C', 1);
		$pdf->Cell($anchoPequeno - 5, 7, 'Desc.', 1, 0, 'C', 1);
		$pdf->Cell($anchoRestante, 7, 'Valor U.', 1, 0, 'C', 1);
		$pdf->Cell($anchoRestante, 7, 'IVA', 1, 0, 'C', 1);
		$pdf->Cell($anchoRestante, 7, 'Importe', 1, 0, 'C', 1);
	}


	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.PublicUnderscore
	/**
	 *  Show top header of page.
	 *
	 *  @param	TCPDF		$pdf     		Object PDF
	 *  @param  Facture		$object     	Object to show
	 *  @param  int	    	$showaddress    0=no, 1=yes
	 *  @param  Translate	$outputlangs	Object lang for output
	 *  @param  Translate	$outputlangsbis	Object lang for output bis
	 *  @return	int							top shift of linked object lines
	 */
	protected function _pagehead(&$pdf, $object, $showaddress, $outputlangs, $outputlangsbis = null, $xml)
	{
		global $conf, $langs, $emisor, $receptor;

		$emisor = getEmisor();
		$emisor['domicilio'] = $conf->global->MAIN_INFO_SOCIETE_ZIP;
		$receptor = getReceptor($object, $object->thirdparty);

		$ltrdirection = 'L';
		if ($outputlangs->trans("DIRECTION") == 'rtl') $ltrdirection = 'R';

		// Load traductions files required by page
		$outputlangs->loadLangs(array("main", "bills", "propal", "companies"));

		$default_font_size = pdf_getPDFFontSize($outputlangs);

		pdf_pagehead($pdf, $outputlangs, $this->page_hauteur);

		$pdf->SetTextColor(0, 0, 60);
		$pdf->SetFont('', 'B', $default_font_size + 3);

		$w = 110;

		$posy = $this->marge_haute;
		$posx = $this->page_largeur - $this->marge_droite - $w;

		$pdf->SetXY($this->marge_gauche, $posy);

		// Logo
		if (empty($conf->global->PDF_DISABLE_MYCOMPANY_LOGO)) {
			if ($this->emetteur->logo) {
				$logodir = $conf->mycompany->dir_output;
				if (!empty($conf->mycompany->multidir_output[$object->entity])) {
					$logodir = $conf->mycompany->multidir_output[$object->entity];
				}
				if (empty($conf->global->MAIN_PDF_USE_LARGE_LOGO)) {
					$logo = $logodir . '/logos/thumbs/' . $this->emetteur->logo_small;
				} else {
					$logo = $logodir . '/logos/' . $this->emetteur->logo;
				}
				if (is_readable($logo)) {
					$height = pdf_getHeightForLogo($logo);
					$pdf->Image($logo, ($this->page_largeur / 2) - $this->marge_gauche - 10, $posy + 5, 0, $height); // width=0 (auto)
				} else {
					$pdf->SetTextColor(200, 0, 0);
					$pdf->SetFont('', 'B', $default_font_size - 2);
					//Show Errors in PDF
					if (!empty($conf->global->CFDIXML_ERROR_PDF)) {

						$pdf->MultiCell($w, 3, $outputlangs->transnoentities("ErrorLogoFileNotFound", $logo), 0, 'L');
						$pdf->MultiCell($w, 3, $outputlangs->transnoentities("ErrorGoToGlobalSetup"), 0, 'L');
					} else {
						$text = $this->emetteur->name;
						$pdf->MultiCell($w, 4, $outputlangs->convToOutputCharset($text), 0, $ltrdirection);
					}
				}
			} else {
				$text = $this->emetteur->name;
				$pdf->MultiCell($w, 4, $outputlangs->convToOutputCharset($text), 0, $ltrdirection);
			}
		}

		$pdf->SetFont('', 'B', $default_font_size + 3);
		$pdf->SetXY($posx, $posy);
		$pdf->SetTextColor(0, 0, 60);
		$title = '';

		switch ($object->type) {

			case Facture::TYPE_CREDIT_NOTE:
				$title = $outputlangs->transnoentities("Egreso");
				break;

			default:
				$title = $outputlangs->transnoentities("Ingreso");
				break;
		}

		$pdf->MultiCell($w, 3, $title, '', 'R');
		$pdf->SetFont('', 'B', $default_font_size);
		$posy += 3;
		$pdf->SetFont('', '', $default_font_size - 2);
		$posy += 4;
		$pdf->SetXY($posx, $posy);
		$pdf->SetTextColor(0, 0, 60);
		$pdf->MultiCell($w, 3, 'Versión 4.0', '', 'R');
		$posy += 4;

		$width = 80;
		$height = 50;
		$radius = 0.5; // Radio de los bordes redondeados

		// Dibujar rectángulo con bordes redondeados
		$pdf->RoundedRect($posx + 30, $posy, $width, $height, $radius, '1111', 'S');
		$posy += 4;
		$posx += 30;
		$width = 79;

		// Preparar texto con manejo seguro de datos XML
		$texto = 'Serie/Folio: ' . $outputlangs->convToOutputCharset($object->ref) . "\n";

		// Validar existencia de datos antes de acceder
		$texto .= 'Fecha: ' . $outputlangs->convToOutputCharset(isset($xml['Fecha']) ? $xml['Fecha'] : 'Pendiente') . "\n";

		// Validar existencia del complemento y timbre fiscal
		if (isset($xml->complemento) && isset($xml->complemento->timbreFiscalDigital)) {
			$texto .= 'Fecha de Timbrado: ' . $outputlangs->convToOutputCharset($xml->complemento->timbreFiscalDigital['fechatimbrado']) . "\n";
			$texto .= 'UUID: ' . $outputlangs->convToOutputCharset($xml->complemento->timbreFiscalDigital['uuid']) . "\n";
			$texto .= 'Certificado SAT: ' . $outputlangs->convToOutputCharset($xml->complemento->timbreFiscalDigital['NoCertificadoSAT']) . "\n";
		} else {
			$texto .= "Fecha de Timbrado: Pendiente\n";
			$texto .= "UUID: Pendiente\n";
			$texto .= "Certificado SAT: Pendiente\n";
		}

		$texto .= 'Certificado: ' . $outputlangs->convToOutputCharset(isset($xml['NoCertificado']) ? $xml['NoCertificado'] : 'Pendiente') . "\n";
		$texto .= 'Método de Pago: ' . $outputlangs->convToOutputCharset(isset($xml['MetodoPago']) ? $xml['MetodoPago'] : 'Pendiente') . "\n";
		$texto .= 'Forma de Pago: ' . $outputlangs->convToOutputCharset(isset($xml['FormaPago']) ? $xml['FormaPago'] : 'Pendiente') . "\n";
		$texto .= 'Condiciones: ' . $outputlangs->convToOutputCharset(isset($xml['CondicionesDePago']) ? $xml['CondicionesDePago'] : 'Pendiente') . "\n";
		$texto .= 'Moneda: ' . $outputlangs->convToOutputCharset(isset($xml['Moneda']) ? $xml['Moneda'] : 'Pendiente') . "\n";


		$lineas = explode("\n", $texto);

		foreach ($lineas as $linea) {
			// Dividir la línea en partes antes y después de ":"
			$partes = explode(':', $linea, 2);

			// Generar las celdas con diferentes alineaciones
			$pdf->setXY($posx, $posy);
			$pdf->MultiCell($width, 5, $partes[0], 0, 'L');

			$pdf->setXY($posx, $posy);
			$pdf->MultiCell($width, 5, $partes[1], 0, 'R');

			// Incrementar posición Y para la siguiente línea
			$posy += 4; // Ajustar según el espaciado deseado entre líneas

		}

		// Show sender
		$posy = !empty($conf->global->MAIN_PDF_USE_ISO_LOCATION) ? 40 : 42;
		$posy += $top_shift;
		$posx = $this->marge_gauche;
		if (!empty($conf->global->MAIN_INVERT_SENDER_RECIPIENT)) {
			$posx = $this->page_largeur - $this->marge_droite - 80;
		}

		$hautcadre = !empty($conf->global->MAIN_PDF_USE_ISO_LOCATION) ? 38 : 40;
		$widthrecbox = !empty($conf->global->MAIN_PDF_USE_ISO_LOCATION) ? 92 : 82;
		$widthrecbox += 25;
		// echo '<pre>';
		// print_r($emisor);
		// exit;
		// Show sender information
		$posy -= 25;
		$pdf->SetFont('', '', $default_font_size + 1);
		$pdf->SetTextColor(0, 0, 0);

		foreach ($emisor as $key => $value) {
			$key = $outputlangs->transnoentities($key);
			$pdf->SetXY($posx, $posy);
			$pdf->MultiCell($widthrecbox, 4, $outputlangs->convToOutputCharset($key) . ': ' . $value, 0, $ltrdirection);
			$posy += 5;
		}
		$posy += 5;
		$endX = 120; // Coordenada X de fin

		$pdf->Line($posx, $posy, $endX, $posy);
		$pdf->SetXY($posx + 50, $posy);
		$pdf->SetTextColor(192, 192, 192);
		$pdf->MultiCell($widthrecbox - 2, 4, "Receptor", 0, $ltrdirection);
		$posy += 4;

		$pdf->SetTextColor(0, 0, 0);
		$pdf->SetFont('', '', $default_font_size);

		foreach ($receptor as $key => $value) {
			$key = $outputlangs->transnoentities($key);
			$pdf->SetXY($posx, $posy);
			$pdf->MultiCell($widthrecbox - 2, 4, $key . ': ' . $value, 0, $ltrdirection);
			$posy += 4;
		}


		$posy += 8;
		$pdf->SetTextColor(0, 0, 0);

		$pdf->SetXY($posx, $posy);

		$pdf->MultiCell($this->page_largeur - $this->marge_droite, 4, "Este documento es una representación impresa de un CFDI v 4.0", 0, 'C');
		return $top_shift;
		// echo '<pre>';print_r($xml->complemento->timbreFiscalDigital['fechatimbrado']);echo '</pre>';exit;
		// echo '<pre>';print_r($xml);echo '</pre>';exit;
	}

	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.PublicUnderscore
	/**
	 *   	Show footer of page. Need this->emetteur object
	 *
	 *   	@param	TCPDF		$pdf     			PDF
	 * 		@param	Facture		$object				Object to show
	 *      @param	Translate	$outputlangs		Object lang for output
	 *      @param	int			$hidefreetext		1=Hide free text
	 *      @return	int								Return height of bottom margin including footer text
	 */
	protected function _pagefoot(&$pdf, $object, $outputlangs, $hidefreetext = 0)
	{
		// 	$showdetails = getDolGlobalInt('MAIN_GENERATE_DOCUMENTS_SHOW_FOOT_DETAILS', 0);
		// 	return pdf_pagefoot($pdf, $outputlangs, 'INVOICE_FREE_TEXT', $this->emetteur, $this->marge_basse, $this->marge_gauche, $this->page_hauteur, $object, $showdetails, $hidefreetext, $this->page_largeur, $this->watermark);
	}



	protected function _cfdixmldata($xml = null)
	{
		$data = array();
		if (!empty($xml)) {

			$xml = base64_decode($xml);
		} else {
			$dir = DOL_DOCUMENT_ROOT;
			if (file_exists(DOL_DOCUMENT_ROOT . '/custom/cfdixml/core/modules/modCfdixml.class.php')) {
				$dir .= '/custom/cfdixml/resources/cfdv4-example.xml';
			} else {
				$dir .= '/cfdixml/resources/cfdv4-example.xml';
			}
			$xml = file_get_contents($dir);
		}
		$cfdi = \CfdiUtils\Cfdi::newFromString($xml);
		// obtener el QuickReader con el método dedicado
		$data = $cfdi->getQuickReader();
		return $data;
	}
}
