<?php
/* Copyright (C) 2017  Laurent Destailleur <eldy@users.sourceforge.net>
 * Copyright (C) 2024 SuperAdmin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

/**
 * \file        class/payment.class.php
 * \ingroup     cfdixml
 * \brief       This file is a CRUD class file for Payment (Create/Read/Update/Delete)
 */

use Symfony\Component\CssSelector\Node\FunctionNode;

// Put here all includes required by your class file
require_once DOL_DOCUMENT_ROOT . '/core/class/commonobject.class.php';
require_once DOL_DOCUMENT_ROOT . '/compta/facture/class/facture.class.php';
require_once DOL_DOCUMENT_ROOT . '/compta/paiement/class/paiement.class.php';
require_once DOL_DOCUMENT_ROOT . '/compta/bank/class/account.class.php';
//require_once DOL_DOCUMENT_ROOT . '/societe/class/societe.class.php';
//require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';

/**
 * Class for Payment
 */
class Payment extends CommonObject
{
	/**
	 * @var string ID of module.
	 */
	public $module = 'cfdixml';

	/**
	 * @var string ID to identify managed object.
	 */
	public $element = 'payment';

	/**
	 * @var string Name of table without prefix where object is stored. This is also the key used for extrafields management.
	 */
	public $table_element = 'cfdixml_payment';

	/**
	 * @var int  Does this object support multicompany module ?
	 * 0=No test on entity, 1=Test with field entity, 'field@table'=Test with link by field@table
	 */
	public $ismultientitymanaged = 0;

	/**
	 * @var int  Does object support extrafields ? 0=No, 1=Yes
	 */
	public $isextrafieldmanaged = 1;

	/**
	 * @var string String with name of icon for payment. Must be a 'fa-xxx' fontawesome code (or 'fa-xxx_fa_color_size') or 'payment@cfdixml' if picto is file 'img/object_payment.png'.
	 */
	public $picto = 'fa-file';


	const STATUS_DRAFT = 0;
	const STATUS_VALIDATED = 1;
	const STATUS_TIMBRADO = 2;
	const STATUS_TIMBRADO_FALLIDO = 8;
	const STATUS_CANCELED = 9;


	/**
	 *  'type' field format:
	 *  	'integer', 'integer:ObjectClass:PathToClass[:AddCreateButtonOrNot[:Filter[:Sortfield]]]',
	 *  	'select' (list of values are in 'options'),
	 *  	'sellist:TableName:LabelFieldName[:KeyFieldName[:KeyFieldParent[:Filter[:Sortfield]]]]',
	 *  	'chkbxlst:...',
	 *  	'varchar(x)',
	 *  	'text', 'text:none', 'html',
	 *   	'double(24,8)', 'real', 'price',
	 *  	'date', 'datetime', 'timestamp', 'duration',
	 *  	'boolean', 'checkbox', 'radio', 'array',
	 *  	'mail', 'phone', 'url', 'password', 'ip'
	 *		Note: Filter must be a Dolibarr filter syntax string. Example: "(t.ref:like:'SO-%') or (t.date_creation:<:'20160101') or (t.status:!=:0) or (t.nature:is:NULL)"
	 *  'label' the translation key.
	 *  'picto' is code of a picto to show before value in forms
	 *  'enabled' is a condition when the field must be managed (Example: 1 or 'getDolGlobalInt('MY_SETUP_PARAM')' or 'isModEnabled("multicurrency")' ...)
	 *  'position' is the sort order of field.
	 *  'notnull' is set to 1 if not null in database. Set to -1 if we must set data to null if empty ('' or 0).
	 *  'visible' says if field is visible in list (Examples: 0=Not visible, 1=Visible on list and create/update/view forms, 2=Visible on list only, 3=Visible on create/update/view form only (not list), 4=Visible on list and update/view form only (not create). 5=Visible on list and view only (not create/not update). Using a negative value means field is not shown by default on list but can be selected for viewing)
	 *  'noteditable' says if field is not editable (1 or 0)
	 *  'alwayseditable' says if field can be modified also when status is not draft ('1' or '0')
	 *  'default' is a default value for creation (can still be overwrote by the Setup of Default Values if field is editable in creation form). Note: If default is set to '(PROV)' and field is 'ref', the default value will be set to '(PROVid)' where id is rowid when a new record is created.
	 *  'index' if we want an index in database.
	 *  'foreignkey'=>'tablename.field' if the field is a foreign key (it is recommanded to name the field fk_...).
	 *  'searchall' is 1 if we want to search in this field when making a search from the quick search button.
	 *  'isameasure' must be set to 1 or 2 if field can be used for measure. Field type must be summable like integer or double(24,8). Use 1 in most cases, or 2 if you don't want to see the column total into list (for example for percentage)
	 *  'css' and 'cssview' and 'csslist' is the CSS style to use on field. 'css' is used in creation and update. 'cssview' is used in view mode. 'csslist' is used for columns in lists. For example: 'css'=>'minwidth300 maxwidth500 widthcentpercentminusx', 'cssview'=>'wordbreak', 'csslist'=>'tdoverflowmax200'
	 *  'help' is a 'TranslationString' to use to show a tooltip on field. You can also use 'TranslationString:keyfortooltiponlick' for a tooltip on click.
	 *  'showoncombobox' if value of the field must be visible into the label of the combobox that list record
	 *  'disabled' is 1 if we want to have the field locked by a 'disabled' attribute. In most cases, this is never set into the definition of $fields into class, but is set dynamically by some part of code.
	 *  'arrayofkeyval' to set a list of values if type is a list of predefined values. For example: array("0"=>"Draft","1"=>"Active","-1"=>"Cancel"). Note that type can be 'integer' or 'varchar'
	 *  'autofocusoncreate' to have field having the focus on a create form. Only 1 field should have this property set to 1.
	 *  'comment' is not used. You can store here any text of your choice. It is not used by application.
	 *	'validate' is 1 if need to validate with $this->validateField()
	 *  'copytoclipboard' is 1 or 2 to allow to add a picto to copy value into clipboard (1=picto after label, 2=picto after value)
	 *
	 *  Note: To have value dynamic, you can set value to 0 in definition and edit the value on the fly into the constructor.
	 */

	// BEGIN MODULEBUILDER PROPERTIES
	/**
	 * @var array  Array with all fields and their property. Do not use it as a static var. It may be modified by constructor.
	 */
	public $fields = array(
		'rowid' => array('type' => 'integer', 'label' => 'TechnicalID', 'enabled' => '1', 'position' => 1, 'notnull' => 1, 'visible' => 0, 'noteditable' => '1', 'index' => 1, 'css' => 'left', 'comment' => "Id"),
		'ref' => array('type' => 'varchar(128)', 'label' => 'Ref', 'enabled' => '1', 'position' => 20, 'notnull' => 1, 'visible' => 4, 'noteditable' => '1', 'default' => '(PROV)', 'index' => 1, 'searchall' => 1, 'showoncombobox' => '1', 'validate' => '1', 'comment' => "Reference of object"),
		'fk_soc' => array('type' => 'integer:Societe:societe/class/societe.class.php:1:((status:=:1) AND (entity:IN:__SHARED_ENTITIES__))', 'label' => 'ThirdParty', 'picto' => 'company', 'enabled' => '$conf->societe->enabled', 'position' => 50, 'notnull' => -1, 'visible' => 1, 'index' => 1, 'css' => 'maxwidth500 widthcentpercentminusxx', 'csslist' => 'tdoverflowmax150', 'help' => "OrganizationEventLinkToThirdParty", 'validate' => '1',),
		'note_public' => array('type' => 'html', 'label' => 'NotePublic', 'enabled' => '1', 'position' => 61, 'notnull' => 0, 'visible' => 0, 'cssview' => 'wordbreak', 'validate' => '1',),
		'note_private' => array('type' => 'html', 'label' => 'NotePrivate', 'enabled' => '1', 'position' => 62, 'notnull' => 0, 'visible' => 0, 'cssview' => 'wordbreak', 'validate' => '1',),
		'date_creation' => array('type' => 'datetime', 'label' => 'DateCreation', 'enabled' => '1', 'position' => 500, 'notnull' => 1, 'visible' => -2,),
		'tms' => array('type' => 'timestamp', 'label' => 'DateModification', 'enabled' => '1', 'position' => 501, 'notnull' => 0, 'visible' => -2,),
		'fk_user_creat' => array('type' => 'integer:User:user/class/user.class.php', 'label' => 'UserAuthor', 'picto' => 'user', 'enabled' => '1', 'position' => 510, 'notnull' => 1, 'visible' => -2, 'foreignkey' => 'user.rowid', 'csslist' => 'tdoverflowmax150',),
		'fk_user_modif' => array('type' => 'integer:User:user/class/user.class.php', 'label' => 'UserModif', 'picto' => 'user', 'enabled' => '1', 'position' => 511, 'notnull' => -1, 'visible' => -2, 'csslist' => 'tdoverflowmax150',),
		'last_main_doc' => array('type' => 'varchar(255)', 'label' => 'LastMainDoc', 'enabled' => '1', 'position' => 600, 'notnull' => 0, 'visible' => 0,),
		'import_key' => array('type' => 'varchar(14)', 'label' => 'ImportId', 'enabled' => '1', 'position' => 1000, 'notnull' => -1, 'visible' => -2,),
		'model_pdf' => array('type' => 'varchar(255)', 'label' => 'Model pdf', 'enabled' => '1', 'position' => 1010, 'notnull' => -1, 'visible' => 0,),
		'status' => array('type' => 'integer', 'label' => 'Status', 'enabled' => '1', 'position' => 2000, 'notnull' => 1, 'visible' => 1, 'default' => '0', 'index' => 1, 'arrayofkeyval' => array('0' => 'Borrador', '1' => 'Validado Sin Timbrar', '8' => 'Timbrado Fallido', '9' => 'Cancelado'), 'validate' => '1',),
		// New fields
		'fechaTimbrado' => array('type' => 'varchar(255)', 'label' => 'Fecha Timbrado', 'enabled' => '1', 'position' => 1020, 'notnull' => 0, 'visible' => 1,),
		'certificado' => array('type' => 'varchar(255)', 'label' => 'Certificado', 'enabled' => '1', 'position' => 1030, 'notnull' => 0, 'visible' => 1,),
		'certificadoSAT' => array('type' => 'varchar(255)', 'label' => 'Certificado SAT', 'enabled' => '1', 'position' => 1040, 'notnull' => 0, 'visible' => 1,),
		'uuid' => array('type' => 'varchar(255)', 'label' => 'UUID', 'enabled' => '1', 'position' => 1050, 'notnull' => 0, 'visible' => 1),
		'xml' => array('type' => 'text', 'label' => 'XML', 'enabled' => '1', 'position' => 1060, 'notnull' => 0, 'visible' => 0,),
	);

	public $rowid;
	public $ref;
	public $fk_soc;
	public $note_public;
	public $note_private;
	public $date_creation;
	public $tms;
	public $fk_user_creat;
	public $fk_user_modif;
	public $last_main_doc;
	public $import_key;
	public $model_pdf;
	public $status;
	public $fechaTimbrado;
	public $certificado;
	public $certificadoSAT;
	public $uuid;
	public $xml;

	// END MODULEBUILDER PROPERTIES


	// If this object has a subtable with lines

	/**
	 * @var string    Name of subtable line
	 */
	public $table_element_line = 'cfdixml_payment_lines';

	/**
	 * @var string    Field with ID of parent key if this object has a parent
	 */
	public $fk_element = 'fk_payment';

	/**
	 * @var string    Name of subtable class that manage subtable lines
	 */
	public $class_element_line = 'Paymentline';

	/**
	 * @var array	List of child tables. To test if we can delete object.
	 */
	protected $childtables = array();

	/**
	 * @var array    List of child tables. To know object to delete on cascade.
	 *               If name matches '@ClassNAme:FilePathClass;ParentFkFieldName' it will
	 *               call method deleteByParentField(parentId, ParentFkFieldName) to fetch and delete child object
	 */
	protected $childtablesoncascade = array('cfdixml_payment_lines');

	/**
	 * @var PaymentLine[]     Array of subtable lines
	 */
	public $lines = array();



	/**
	 * Constructor
	 *
	 * @param DoliDb $db Database handler
	 */
	public function __construct(DoliDB $db)
	{
		global $conf, $langs;

		$this->db = $db;

		if (!getDolGlobalInt('MAIN_SHOW_TECHNICAL_ID') && isset($this->fields['rowid']) && !empty($this->fields['ref'])) {
			$this->fields['rowid']['visible'] = 0;
		}
		if (!isModEnabled('multicompany') && isset($this->fields['entity'])) {
			$this->fields['entity']['enabled'] = 0;
		}

		// Example to show how to set values of fields definition dynamically
		/*if ($user->rights->cfdixml->payment->read) {
			$this->fields['myfield']['visible'] = 1;
			$this->fields['myfield']['noteditable'] = 0;
		}*/

		// Unset fields that are disabled
		foreach ($this->fields as $key => $val) {
			if (isset($val['enabled']) && empty($val['enabled'])) {
				unset($this->fields[$key]);
			}
		}

		// Translate some data of arrayofkeyval
		if (is_object($langs)) {
			foreach ($this->fields as $key => $val) {
				if (!empty($val['arrayofkeyval']) && is_array($val['arrayofkeyval'])) {
					foreach ($val['arrayofkeyval'] as $key2 => $val2) {
						$this->fields[$key]['arrayofkeyval'][$key2] = $langs->trans($val2);
					}
				}
			}
		}
	}

	/**
	 * Returns a list of payments made by a specific third-party.
	 *
	 * @param int $socid The ID of the third-party.
	 * @return array|int An array of payment objects, or -1 if an error occurred.
	 */

	public function getPayments($socid, $paymentid = 0)
	{
		$sql = "SELECT p.rowid as pay_rowid, p.datec as date_payment, p.ref as pay_ref,p.fk_paiement as code_payment ,f.rowid as fact_id, f.ref as fact_ref, f.multicurrency_total_ttc as fact_total ";
		$sql .= ",f.multicurrency_code as code, pf.amount as pay_amount, pf.multicurrency_amount as pay_total, pf.rowid as payfact_rowid ";

		// Agregar condicional para obtener el campo 'code' de 'llx_cfdixml_payment_invoice' si 'multicurrency_code' es diferente de 'MX'
		$sql .= ",CASE WHEN f.multicurrency_code <> 'MX' THEN c.code ELSE NULL END as payment_code";
		$sql .= ",CASE WHEN f.multicurrency_code <> 'MX' THEN c.tx ELSE NULL END as payment_tx";

		$sql .= " FROM " . MAIN_DB_PREFIX . "paiement p ";
		$sql .= " LEFT JOIN " . MAIN_DB_PREFIX . "paiement_extrafields pe ON p.rowid = pe.fk_object ";
		$sql .= " LEFT JOIN " . MAIN_DB_PREFIX . "paiement_facture pf ON p.rowid = pf.fk_paiement ";
		$sql .= " LEFT JOIN " . MAIN_DB_PREFIX . "facture f ON f.rowid = pf.fk_facture ";
		$sql .= " LEFT JOIN " . MAIN_DB_PREFIX . "facture_extrafields fe ON f.rowid = fe.fk_object ";

		// Si 'multicurrency_code' es diferente de 'MX', unir con 'llx_cfdixml_payment_invoice' para obtener 'code'
		$sql .= " LEFT JOIN " . MAIN_DB_PREFIX . "cfdixml_payment_invoice c ON c.fk_facture = f.rowid AND f.multicurrency_code <> 'MX'";
		if ($paymentid) {
			$sql .= " LEFT JOIN " . MAIN_DB_PREFIX . "cfdixml_payment_lines cpl ON cpl.fk_paiement = p.rowid";
		}

		$sql .= " WHERE f.fk_soc = " . $socid;
		$sql .= " AND fe.cfdixml_metodopago = 'PPD' AND fe.cfdixml_UUID IS NOT NULL";
		$sql .= " AND pe.cfdixml_UUID IS NULL";
		if ($paymentid) {
			$sql .= " AND NOT EXISTS (
				SELECT 1
				FROM llx_cfdixml_payment_lines cpl
				WHERE cpl.fk_paiement = p.rowid
				AND cpl.fk_payment = {$paymentid}
			)";
		}
		$sql .= " ORDER BY p.datec DESC";
		// echo $sql;
		$resql = $this->db->query($sql);
		if ($resql) {
			$payments = array();
			while ($row = $this->db->fetch_object($resql)) {
				$payments[] = $row;
			}
			$this->db->free($resql);

			return $payments;
		}
	}

	function getCodeFromId($id, $table, $label = false, $libelle = false)
	{

		$code = "code";
		if ($label) $code = "label";
		if ($libelle) $code = "libelle";
		$sql = "SELECT {$code} FROM " . MAIN_DB_PREFIX . $table . " WHERE rowid = " . $id;
		$resql = $this->db->query($sql);
		if ($resql) {
			$row = $this->db->fetch_object($resql);
			$this->db->free($resql);
			return $row->$code;
		} else {
			$sql = "SELECT {$code} FROM " . MAIN_DB_PREFIX . $table . " WHERE id = " . $id;

			$resql = $this->db->query($sql);
			if ($resql) {
				$row = $this->db->fetch_object($resql);
				$this->db->free($resql);
				return $row->$code;
			}
		}
	}

	/**
	 * Create object into database
	 *
	 * @param  User $user      User that creates
	 * @param  bool $notrigger false=launch triggers after, true=disable triggers
	 * @return int             <0 if KO, Id of created object if OK
	 */
	public function create(User $user, $notrigger = false)
	{

		$resultcreate = $this->createCommon($user, $notrigger);

		//$resultvalidate = $this->validate($user, $notrigger);

		return $resultcreate;
	}

	/**
	 * Clone an object into another one
	 *
	 * @param  	User 	$user      	User that creates
	 * @param  	int 	$fromid     Id of object to clone
	 * @return 	mixed 				New object created, <0 if KO
	 */
	public function createFromClone(User $user, $fromid)
	{
		global $langs, $extrafields;
		$error = 0;

		dol_syslog(__METHOD__, LOG_DEBUG);

		$object = new self($this->db);

		$this->db->begin();

		// Load source object
		$result = $object->fetchCommon($fromid);
		if ($result > 0 && !empty($object->table_element_line)) {
			$object->fetchLines();
		}

		// get lines so they will be clone
		//foreach($this->lines as $line)
		//	$line->fetch_optionals();

		// Reset some properties
		unset($object->id);
		unset($object->fk_user_creat);
		unset($object->import_key);

		// Clear fields
		if (property_exists($object, 'ref')) {
			$object->ref = empty($this->fields['ref']['default']) ? "Copy_Of_" . $object->ref : $this->fields['ref']['default'];
		}
		if (property_exists($object, 'label')) {
			$object->label = empty($this->fields['label']['default']) ? $langs->trans("CopyOf") . " " . $object->label : $this->fields['label']['default'];
		}
		if (property_exists($object, 'status')) {
			$object->status = self::STATUS_DRAFT;
		}
		if (property_exists($object, 'date_creation')) {
			$object->date_creation = dol_now();
		}
		if (property_exists($object, 'date_modification')) {
			$object->date_modification = null;
		}
		// ...
		// Clear extrafields that are unique
		if (is_array($object->array_options) && count($object->array_options) > 0) {
			$extrafields->fetch_name_optionals_label($this->table_element);
			foreach ($object->array_options as $key => $option) {
				$shortkey = preg_replace('/options_/', '', $key);
				if (!empty($extrafields->attributes[$this->table_element]['unique'][$shortkey])) {
					//var_dump($key);
					//var_dump($clonedObj->array_options[$key]); exit;
					unset($object->array_options[$key]);
				}
			}
		}

		// Create clone
		$object->context['createfromclone'] = 'createfromclone';
		$result = $object->createCommon($user);
		if ($result < 0) {
			$error++;
			$this->error = $object->error;
			$this->errors = $object->errors;
		}

		if (!$error) {
			// copy internal contacts
			if ($this->copy_linked_contact($object, 'internal') < 0) {
				$error++;
			}
		}

		if (!$error) {
			// copy external contacts if same company
			if (!empty($object->socid) && property_exists($this, 'fk_soc') && $this->fk_soc == $object->socid) {
				if ($this->copy_linked_contact($object, 'external') < 0) {
					$error++;
				}
			}
		}

		unset($object->context['createfromclone']);

		// End
		if (!$error) {
			$this->db->commit();
			return $object;
		} else {
			$this->db->rollback();
			return -1;
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param int    $id   Id object
	 * @param string $ref  Ref
	 * @return int         <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetch($id, $ref = null)
	{
		$result = $this->fetchCommon($id, $ref);
		if ($result > 0 && !empty($this->table_element_line)) {
			$this->fetchLines();
		}
		return $result;
	}

	/**
	 * Load object lines in memory from the database
	 *
	 * @return int         <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetchLines()
	{
		$this->lines = array();

		$objectline = new PaymentLine($this->db);
		$result = $objectline->fetchLines(((int) $this->id));
		return $result;
	}


	/**
	 * Load list of objects in memory from the database.
	 *
	 * @param  string      $sortorder    Sort Order
	 * @param  string      $sortfield    Sort field
	 * @param  int         $limit        limit
	 * @param  int         $offset       Offset
	 * @param  array       $filter       Filter array. Example array('field'=>'valueforlike', 'customurl'=>...)
	 * @param  string      $filtermode   Filter mode (AND or OR)
	 * @return array|int                 int <0 if KO, array of pages if OK
	 */
	public function fetchAll($sortorder = '', $sortfield = '', $limit = 0, $offset = 0, array $filter = array(), $filtermode = 'AND')
	{
		global $conf;

		dol_syslog(__METHOD__, LOG_DEBUG);

		$records = array();

		$sql = "SELECT ";
		$sql .= $this->getFieldList('t');
		$sql .= " FROM " . $this->db->prefix() . $this->table_element . " as t";
		if (isset($this->ismultientitymanaged) && $this->ismultientitymanaged == 1) {
			$sql .= " WHERE t.entity IN (" . getEntity($this->element) . ")";
		} else {
			$sql .= " WHERE 1 = 1";
		}
		// Manage filter
		$sqlwhere = array();
		if (count($filter) > 0) {
			foreach ($filter as $key => $value) {
				if ($key == 't.rowid') {
					$sqlwhere[] = $key . " = " . ((int) $value);
				} elseif (in_array($this->fields[$key]['type'], array('date', 'datetime', 'timestamp'))) {
					$sqlwhere[] = $key . " = '" . $this->db->idate($value) . "'";
				} elseif ($key == 'customsql') {
					$sqlwhere[] = $value;
				} elseif (strpos($value, '%') === false) {
					$sqlwhere[] = $key . " IN (" . $this->db->sanitize($this->db->escape($value)) . ")";
				} else {
					$sqlwhere[] = $key . " LIKE '%" . $this->db->escape($value) . "%'";
				}
			}
		}
		if (count($sqlwhere) > 0) {
			$sql .= " AND (" . implode(" " . $filtermode . " ", $sqlwhere) . ")";
		}

		if (!empty($sortfield)) {
			$sql .= $this->db->order($sortfield, $sortorder);
		}
		if (!empty($limit)) {
			$sql .= $this->db->plimit($limit, $offset);
		}

		$resql = $this->db->query($sql);
		if ($resql) {
			$num = $this->db->num_rows($resql);
			$i = 0;
			while ($i < ($limit ? min($limit, $num) : $num)) {
				$obj = $this->db->fetch_object($resql);

				$record = new self($this->db);
				$record->setVarsFromFetchObj($obj);

				$records[$record->id] = $record;

				$i++;
			}
			$this->db->free($resql);

			return $records;
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
	 * Update object into database
	 *
	 * @param  User $user      User that modifies
	 * @param  bool $notrigger false=launch triggers after, true=disable triggers
	 * @return int             <0 if KO, >0 if OK
	 */
	public function update(User $user, $notrigger = false)
	{
		return $this->updateCommon($user, $notrigger);
	}

	/**
	 * Delete object in database
	 *
	 * @param User $user       User that deletes
	 * @param bool $notrigger  false=launch triggers, true=disable triggers
	 * @return int             <0 if KO, >0 if OK
	 */
	public function delete(User $user, $notrigger = false)
	{
		return $this->deleteCommon($user, $notrigger);
		//return $this->deleteCommon($user, $notrigger, 1);
	}

	/**
	 *  Delete a line of object in database
	 *
	 *	@param  User	$user       User that delete
	 *  @param	int		$idline		Id of line to delete
	 *  @param 	bool 	$notrigger  false=launch triggers after, true=disable triggers
	 *  @return int         		>0 if OK, <0 if KO
	 */
	public function deleteLine(User $user, $idline, $notrigger = false)
	{
		if ($this->status < 0) {
			$this->error = 'ErrorDeleteLineNotAllowedByObjectStatus';
			return -2;
		}

		return $this->deleteLineCommon($user, $idline, $notrigger);
	}


	/**
	 *	Validate object
	 *
	 *	@param		User	$user     		User making status change
	 *  @param		int		$notrigger		1=Does not execute triggers, 0= execute triggers
	 *	@return  	int						<=0 if OK, 0=Nothing done, >0 if KO
	 */
	public function validate($user, $notrigger = 0)
	{
		global $conf, $langs;

		require_once DOL_DOCUMENT_ROOT . '/core/lib/files.lib.php';

		$error = 0;

		// Protection
		if ($this->status == self::STATUS_VALIDATED) {
			dol_syslog(get_class($this) . "::validate action abandonned: already validated", LOG_WARNING);
			return 0;
		}

		/* if (! ((!getDolGlobalInt('MAIN_USE_ADVANCED_PERMS') && $user->hasRight('cfdixml','write'))
		 || (getDolGlobalInt('MAIN_USE_ADVANCED_PERMS') && !empty($user->rights->cfdixml->payment->payment_advance->validate))))
		 {
		 $this->error='NotEnoughPermissions';
		 dol_syslog(get_class($this)."::valid ".$this->error, LOG_ERR);
		 return -1;
		 }*/

		$now = dol_now();

		$this->db->begin();

		// Define new ref
		if (!$error && (preg_match('/^[\(]?PROV/i', $this->ref) || empty($this->ref))) { // empty should not happened, but when it occurs, the test save life
			$num = $this->getNextNumRef();
		} else {
			$num = $this->ref;
		}
		$this->newref = $num;

		if (!empty($num)) {
			// Validate
			$sql = "UPDATE " . MAIN_DB_PREFIX . $this->table_element;
			$sql .= " SET ref = '" . $this->db->escape($num) . "',";
			$sql .= " status = " . self::STATUS_VALIDATED;
			if (!empty($this->fields['date_validation'])) {
				$sql .= ", date_validation = '" . $this->db->idate($now) . "'";
			}
			if (!empty($this->fields['fk_user_valid'])) {
				$sql .= ", fk_user_valid = " . ((int) $user->id);
			}
			$sql .= " WHERE rowid = " . ((int) $this->id);

			dol_syslog(get_class($this) . "::validate()", LOG_DEBUG);
			$resql = $this->db->query($sql);
			if (!$resql) {
				dol_print_error($this->db);
				$this->error = $this->db->lasterror();
				$error++;
			}

			if (!$error && !$notrigger) {
				// Call trigger
				$result = $this->call_trigger('PAYMENT_VALIDATE', $user);
				if ($result < 0) {
					$error++;
				}
				// End call triggers
			}
		}

		if (!$error) {
			$this->oldref = $this->ref;

			// Rename directory if dir was a temporary ref
			if (preg_match('/^[\(]?PROV/i', $this->ref)) {
				// Now we rename also files into index
				$sql = 'UPDATE ' . MAIN_DB_PREFIX . "ecm_files set filename = CONCAT('" . $this->db->escape($this->newref) . "', SUBSTR(filename, " . (strlen($this->ref) + 1) . ")), filepath = 'payment/" . $this->db->escape($this->newref) . "'";
				$sql .= " WHERE filename LIKE '" . $this->db->escape($this->ref) . "%' AND filepath = 'payment/" . $this->db->escape($this->ref) . "' and entity = " . $conf->entity;
				$resql = $this->db->query($sql);
				if (!$resql) {
					$error++;
					$this->error = $this->db->lasterror();
				}

				// We rename directory ($this->ref = old ref, $num = new ref) in order not to lose the attachments
				$oldref = dol_sanitizeFileName($this->ref);
				$newref = dol_sanitizeFileName($num);
				$dirsource = $conf->cfdixml->dir_output . '/payment/' . $oldref;
				$dirdest = $conf->cfdixml->dir_output . '/payment/' . $newref;
				if (!$error && file_exists($dirsource)) {
					dol_syslog(get_class($this) . "::validate() rename dir " . $dirsource . " into " . $dirdest);

					if (@rename($dirsource, $dirdest)) {
						dol_syslog("Rename ok");
						// Rename docs starting with $oldref with $newref
						$listoffiles = dol_dir_list($conf->cfdixml->dir_output . '/payment/' . $newref, 'files', 1, '^' . preg_quote($oldref, '/'));
						foreach ($listoffiles as $fileentry) {
							$dirsource = $fileentry['name'];
							$dirdest = preg_replace('/^' . preg_quote($oldref, '/') . '/', $newref, $dirsource);
							$dirsource = $fileentry['path'] . '/' . $dirsource;
							$dirdest = $fileentry['path'] . '/' . $dirdest;
							@rename($dirsource, $dirdest);
						}
					}
				}
			}
		}

		// Set new ref and current status
		if (!$error) {
			$this->ref = $num;
			$this->status = self::STATUS_VALIDATED;
		}

		if (!$error) {
			$this->db->commit();
			return 1;
		} else {
			$this->db->rollback();
			return -1;
		}
	}


	/**
	 *	Set draft status
	 *
	 *	@param	User	$user			Object user that modify
	 *  @param	int		$notrigger		1=Does not execute triggers, 0=Execute triggers
	 *	@return	int						<0 if KO, >0 if OK
	 */
	public function setDraft($user, $notrigger = 0)
	{
		// Protection
		if ($this->status <= self::STATUS_DRAFT) {
			return 0;
		}

		/* if (! ((!getDolGlobalInt('MAIN_USE_ADVANCED_PERMS') && $user->hasRight('cfdixml','write'))
		 || (getDolGlobalInt('MAIN_USE_ADVANCED_PERMS') && $user->hasRight('cfdixml','cfdixml_advance','validate'))))
		 {
		 $this->error='Permission denied';
		 return -1;
		 }*/

		return $this->setStatusCommon($user, self::STATUS_DRAFT, $notrigger, 'PAYMENT_UNVALIDATE');
	}

	/**
	 *	Set cancel status
	 *
	 *	@param	User	$user			Object user that modify
	 *  @param	int		$notrigger		1=Does not execute triggers, 0=Execute triggers
	 *	@return	int						<0 if KO, 0=Nothing done, >0 if OK
	 */
	public function cancel($user, $notrigger = 0)
	{
		// Protection
		if ($this->status != self::STATUS_VALIDATED) {
			return 0;
		}

		/* if (! ((!getDolGlobalInt('MAIN_USE_ADVANCED_PERMS') && $user->hasRight('cfdixml','write'))
		 || (getDolGlobalInt('MAIN_USE_ADVANCED_PERMS') && $user->hasRight('cfdixml','cfdixml_advance','validate'))))
		 {
		 $this->error='Permission denied';
		 return -1;
		 }*/

		return $this->setStatusCommon($user, self::STATUS_CANCELED, $notrigger, 'PAYMENT_CANCEL');
	}

	/**
	 *	Set back to validated status
	 *
	 *	@param	User	$user			Object user that modify
	 *  @param	int		$notrigger		1=Does not execute triggers, 0=Execute triggers
	 *	@return	int						<0 if KO, 0=Nothing done, >0 if OK
	 */
	public function reopen($user, $notrigger = 0)
	{
		// Protection
		if ($this->status == self::STATUS_VALIDATED) {
			return 0;
		}

		/*if (! ((!getDolGlobalInt('MAIN_USE_ADVANCED_PERMS') && $user->hasRight('cfdixml','write'))
		 || (getDolGlobalInt('MAIN_USE_ADVANCED_PERMS') && $user->hasRight('cfdixml','cfdixml_advance','validate'))))
		 {
		 $this->error='Permission denied';
		 return -1;
		 }*/


		return $this->setStatusCommon($user, self::STATUS_VALIDATED, $notrigger, 'PAYMENT_REOPEN');
	}

	/**
	 *  Return a link to the object card (with optionaly the picto)
	 *
	 *  @param  int     $withpicto                  Include picto in link (0=No picto, 1=Include picto into link, 2=Only picto)
	 *  @param  string  $option                     On what the link point to ('nolink', ...)
	 *  @param  int     $notooltip                  1=Disable tooltip
	 *  @param  string  $morecss                    Add more css on link
	 *  @param  int     $save_lastsearch_value      -1=Auto, 0=No save of lastsearch_values when clicking, 1=Save lastsearch_values whenclicking
	 *  @return	string                              String with URL
	 */
	public function getNomUrl($withpicto = 0, $option = '', $notooltip = 0, $morecss = '', $save_lastsearch_value = -1)
	{
		global $conf, $langs, $hookmanager;

		if (!empty($conf->dol_no_mouse_hover)) {
			$notooltip = 1; // Force disable tooltips
		}

		$result = '';

		$label = img_picto('', $this->picto) . ' <u>' . $langs->trans("Payment") . '</u>';
		if (isset($this->status)) {
			$label .= ' ' . $this->getLibStatut(5);
		}
		$label .= '<br>';
		$label .= '<b>' . $langs->trans('Ref') . ':</b> ' . $this->ref;

		$url = dol_buildpath('/cfdixml/payments/payment_card.php', 1) . '?id=' . $this->id;

		if ($option != 'nolink') {
			// Add param to save lastsearch_values or not
			$add_save_lastsearch_values = ($save_lastsearch_value == 1 ? 1 : 0);
			if ($save_lastsearch_value == -1 && preg_match('/list\.php/', $_SERVER["PHP_SELF"])) {
				$add_save_lastsearch_values = 1;
			}
			if ($url && $add_save_lastsearch_values) {
				$url .= '&save_lastsearch_values=1';
			}
		}

		$linkclose = '';
		if (empty($notooltip)) {
			if (getDolGlobalInt('MAIN_OPTIMIZEFORTEXTBROWSER')) {
				$label = $langs->trans("ShowPayment");
				$linkclose .= ' alt="' . dol_escape_htmltag($label, 1) . '"';
			}
			$linkclose .= ' title="' . dol_escape_htmltag($label, 1) . '"';
			$linkclose .= ' class="classfortooltip' . ($morecss ? ' ' . $morecss : '') . '"';
		} else {
			$linkclose = ($morecss ? ' class="' . $morecss . '"' : '');
		}

		if ($option == 'nolink' || empty($url)) {
			$linkstart = '<span';
		} else {
			$linkstart = '<a href="' . $url . '"';
		}
		$linkstart .= $linkclose . '>';
		if ($option == 'nolink' || empty($url)) {
			$linkend = '</span>';
		} else {
			$linkend = '</a>';
		}

		$result .= $linkstart;

		if (empty($this->showphoto_on_popup)) {
			if ($withpicto) {
				$result .= img_object(($notooltip ? '' : $label), ($this->picto ? $this->picto : 'generic'), ($notooltip ? (($withpicto != 2) ? 'class="paddingright"' : '') : 'class="' . (($withpicto != 2) ? 'paddingright ' : '') . 'classfortooltip"'), 0, 0, $notooltip ? 0 : 1);
			}
		} else {
			if ($withpicto) {
				require_once DOL_DOCUMENT_ROOT . '/core/lib/files.lib.php';

				list($class, $module) = explode('@', $this->picto);
				$upload_dir = $conf->$module->multidir_output[$conf->entity] . "/$class/" . dol_sanitizeFileName($this->ref);
				$filearray = dol_dir_list($upload_dir, "files");
				$filename = $filearray[0]['name'];
				if (!empty($filename)) {
					$pospoint = strpos($filearray[0]['name'], '.');

					$pathtophoto = $class . '/' . $this->ref . '/thumbs/' . substr($filename, 0, $pospoint) . '_mini' . substr($filename, $pospoint);
					if (!getDolGlobalString(strtoupper($module . '_' . $class) . '_FORMATLISTPHOTOSASUSERS')) {
						$result .= '<div class="floatleft inline-block valignmiddle divphotoref"><div class="photoref"><img class="photo' . $module . '" alt="No photo" border="0" src="' . DOL_URL_ROOT . '/viewimage.php?modulepart=' . $module . '&entity=' . $conf->entity . '&file=' . urlencode($pathtophoto) . '"></div></div>';
					} else {
						$result .= '<div class="floatleft inline-block valignmiddle divphotoref"><img class="photouserphoto userphoto" alt="No photo" border="0" src="' . DOL_URL_ROOT . '/viewimage.php?modulepart=' . $module . '&entity=' . $conf->entity . '&file=' . urlencode($pathtophoto) . '"></div>';
					}

					$result .= '</div>';
				} else {
					$result .= img_object(($notooltip ? '' : $label), ($this->picto ? $this->picto : 'generic'), ($notooltip ? (($withpicto != 2) ? 'class="paddingright"' : '') : 'class="' . (($withpicto != 2) ? 'paddingright ' : '') . 'classfortooltip"'), 0, 0, $notooltip ? 0 : 1);
				}
			}
		}

		if ($withpicto != 2) {
			$result .= $this->ref;
		}

		$result .= $linkend;
		//if ($withpicto != 2) $result.=(($addlabel && $this->label) ? $sep . dol_trunc($this->label, ($addlabel > 1 ? $addlabel : 0)) : '');

		global $action, $hookmanager;
		$hookmanager->initHooks(array($this->element . 'dao'));
		$parameters = array('id' => $this->id, 'getnomurl' => &$result);
		$reshook = $hookmanager->executeHooks('getNomUrl', $parameters, $this, $action); // Note that $action and $object may have been modified by some hooks
		if ($reshook > 0) {
			$result = $hookmanager->resPrint;
		} else {
			$result .= $hookmanager->resPrint;
		}

		return $result;
	}

	/**
	 *	Return a thumb for kanban views
	 *
	 *	@param      string	    $option                 Where point the link (0=> main card, 1,2 => shipment, 'nolink'=>No link)
	 *  @return		string								HTML Code for Kanban thumb.
	 */
	/*
	public function getKanbanView($option = '')
	{
		$return = '<div class="box-flex-item box-flex-grow-zero">';
		$return .= '<div class="info-box info-box-sm">';
		$return .= '<span class="info-box-icon bg-infobox-action">';
		$return .= img_picto('', $this->picto);
		$return .= '</span>';
		$return .= '<div class="info-box-content">';
		$return .= '<span class="info-box-ref">'.(method_exists($this, 'getNomUrl') ? $this->getNomUrl() : $this->ref).'</span>';
		if (property_exists($this, 'label')) {
			$return .= '<br><span class="info-box-label opacitymedium">'.$this->label.'</span>';
		}
		if (method_exists($this, 'getLibStatut')) {
			$return .= '<br><div class="info-box-status margintoponly">'.$this->getLibStatut(5).'</div>';
		}
		$return .= '</div>';
		$return .= '</div>';
		$return .= '</div>';

		return $return;
	}
	*/

	/**
	 *  Return the label of the status
	 *
	 *  @param  int		$mode          0=long label, 1=short label, 2=Picto + short label, 3=Picto, 4=Picto + long label, 5=Short label + Picto, 6=Long label + Picto
	 *  @return	string 			       Label of status
	 */
	public function getLabelStatus($mode = 0)
	{
		return $this->LibStatut($this->status, $mode);
	}

	/**
	 *  Return the label of the status
	 *
	 *  @param  int		$mode          0=long label, 1=short label, 2=Picto + short label, 3=Picto, 4=Picto + long label, 5=Short label + Picto, 6=Long label + Picto
	 *  @return	string 			       Label of status
	 */
	public function getLibStatut($mode = 0)
	{
		return $this->LibStatut($this->status, $mode);
	}

	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
	/**
	 *  Return the status
	 *
	 *  @param	int		$status        Id status
	 *  @param  int		$mode          0=long label, 1=short label, 2=Picto + short label, 3=Picto, 4=Picto + long label, 5=Short label + Picto, 6=Long label + Picto
	 *  @return string 			       Label of status
	 */
	public function LibStatut($status, $mode = 0)
	{
		// phpcs:enable
		if (empty($this->labelStatus) || empty($this->labelStatusShort)) {
			global $langs;
			//$langs->load("cfdixml@cfdixml");
			$this->labelStatus[self::STATUS_DRAFT] = $langs->transnoentitiesnoconv('Draft');
			$this->labelStatus[self::STATUS_VALIDATED] = $langs->transnoentitiesnoconv('EnabledPayCfdixml');
			$this->labelStatus[self::STATUS_TIMBRADO] = $langs->transnoentitiesnoconv('SuccessStamped');
			$this->labelStatus[self::STATUS_TIMBRADO_FALLIDO] = $langs->transnoentitiesnoconv('FailedledPayCfdixml');
			$this->labelStatus[self::STATUS_CANCELED] = $langs->transnoentitiesnoconv('Disabled');
			$this->labelStatusShort[self::STATUS_DRAFT] = $langs->transnoentitiesnoconv('Draft');
			$this->labelStatusShort[self::STATUS_VALIDATED] = $langs->transnoentitiesnoconv('EnabledPayCfdixml');
			$this->labelStatusShort[self::STATUS_TIMBRADO] = $langs->transnoentitiesnoconv('SuccessStamped');
			$this->labelStatusShort[self::STATUS_TIMBRADO_FALLIDO] = $langs->transnoentitiesnoconv('FailedledPayCfdixml');
			$this->labelStatusShort[self::STATUS_CANCELED] = $langs->transnoentitiesnoconv('Disabled');
		}

		$statusType = 'status' . $status;
		//if ($status == self::STATUS_VALIDATED) $statusType = 'status1';
		if ($status == self::STATUS_CANCELED) {
			$statusType = 'status6';
		}

		return dolGetStatus($this->labelStatus[$status], $this->labelStatusShort[$status], '', $statusType, $mode);
	}

	/**
	 *	Load the info information in the object
	 *
	 *	@param  int		$id       Id of object
	 *	@return	void
	 */
	public function info($id)
	{
		$sql = "SELECT rowid,";
		$sql .= " date_creation as datec, tms as datem,";
		$sql .= " fk_user_creat, fk_user_modif";
		$sql .= " FROM " . MAIN_DB_PREFIX . $this->table_element . " as t";
		$sql .= " WHERE t.rowid = " . ((int) $id);

		$result = $this->db->query($sql);
		if ($result) {
			if ($this->db->num_rows($result)) {
				$obj = $this->db->fetch_object($result);

				$this->id = $obj->rowid;

				$this->user_creation_id = $obj->fk_user_creat;
				$this->user_modification_id = $obj->fk_user_modif;
				if (!empty($obj->fk_user_valid)) {
					$this->user_validation_id = $obj->fk_user_valid;
				}
				$this->date_creation     = $this->db->jdate($obj->datec);
				$this->date_modification = empty($obj->datem) ? '' : $this->db->jdate($obj->datem);
				if (!empty($obj->datev)) {
					$this->date_validation   = empty($obj->datev) ? '' : $this->db->jdate($obj->datev);
				}
			}

			$this->db->free($result);
		} else {
			dol_print_error($this->db);
		}
	}

	/**
	 * Initialise object with example values
	 * Id must be 0 if object instance is a specimen
	 *
	 * @return void
	 */
	public function initAsSpecimen()
	{
		// Set here init that are not commonf fields
		// $this->property1 = ...
		// $this->property2 = ...

		$this->initAsSpecimenCommon();
	}

	/**
	 * 	Create an array of lines
	 *
	 * 	@return array|int		array of lines if OK, <0 if KO
	 */
	public function getLinesArray()
	{
		$this->lines = array();

		$objectline = new PaymentLine($this->db);
		$result = $objectline->fetchLines(((int) $this->id));

		if (is_numeric($result)) {
			$this->error = $objectline->error;
			$this->errors = $objectline->errors;
			return $result;
		} else {
			$this->lines = $result;
			return $this->lines;
		}
	}

	/**
	 *  Returns the reference to the following non used object depending on the active numbering module.
	 *
	 *  @return string      		Object free reference
	 */
	public function getNextNumRef()
	{
		global $langs, $conf;
		$langs->load("cfdixml@cfdixml");

		if (getDolGlobalString('CFDIXML_PAYMENT_ADDON') == '') {
			$conf->global->CFDIXML_PAYMENT_ADDON = 'mod_payment_standard';
		}

		if (getDolGlobalString('CFDIXML_PAYMENT_ADDON')) {
			$mybool = false;

			$file = getDolGlobalString('CFDIXML_PAYMENT_ADDON') . ".php";
			$classname = getDolGlobalString('CFDIXML_PAYMENT_ADDON');

			// Include file with class
			$dirmodels = array_merge(array('/'), (array) $conf->modules_parts['models']);
			foreach ($dirmodels as $reldir) {
				$dir = dol_buildpath($reldir . "core/modules/cfdixml/");

				// Load file with numbering class (if found)
				$mybool |= @include_once $dir . $file;
			}

			if ($mybool === false) {
				dol_print_error('', "Failed to include file " . $file);
				return '';
			}

			if (class_exists($classname)) {
				$obj = new $classname();
				$numref = $obj->getNextValue($this);

				if ($numref != '' && $numref != '-1') {
					return $numref;
				} else {
					$this->error = $obj->error;
					//dol_print_error($this->db,get_class($this)."::getNextNumRef ".$obj->error);
					return "";
				}
			} else {
				print $langs->trans("Error") . " " . $langs->trans("ClassNotFound") . ' ' . $classname;
				return "";
			}
		} else {
			print $langs->trans("ErrorNumberingModuleNotSetup", $this->element);
			return "";
		}
	}

	/**
	 *  Create a document onto disk according to template module.
	 *
	 *  @param	    string		$modele			Force template to use ('' to not force)
	 *  @param		Translate	$outputlangs	objet lang a utiliser pour traduction
	 *  @param      int			$hidedetails    Hide details of lines
	 *  @param      int			$hidedesc       Hide description
	 *  @param      int			$hideref        Hide ref
	 *  @param      null|array  $moreparams     Array to provide more information
	 *  @return     int         				0 if KO, 1 if OK
	 */
	public function generateDocument($modele, $outputlangs, $hidedetails = 0, $hidedesc = 0, $hideref = 0, $moreparams = null)
	{
		global $conf, $langs;

		$result = 0;
		$includedocgeneration = 1;

		$langs->load("cfdixml@cfdixml");

		if (!dol_strlen($modele)) {
			$modele = 'standard_payment';

			if (!empty($this->model_pdf)) {
				$modele = $this->model_pdf;
			} elseif (getDolGlobalString('PAYMENT_ADDON_PDF')) {
				$modele = getDolGlobalString('PAYMENT_ADDON_PDF');
			}
		}

		$modelpath = "core/modules/cfdixml/doc/";

		if ($includedocgeneration && !empty($modele)) {
			$result = $this->commonGenerateDocument($modelpath, $modele, $outputlangs, $hidedetails, $hidedesc, $hideref, $moreparams);
		}

		return $result;
	}

	/**
	 * Action executed by scheduler
	 * CAN BE A CRON TASK. In such a case, parameters come from the schedule job setup field 'Parameters'
	 * Use public function doScheduledJob($param1, $param2, ...) to get parameters
	 *
	 * @return	int			0 if OK, <>0 if KO (this function is used also by cron so only 0 is OK)
	 */
	public function doScheduledJob()
	{
		global $conf, $langs;

		//$conf->global->SYSLOG_FILE = 'DOL_DATA_ROOT/dolibarr_mydedicatedlofile.log';

		$error = 0;
		$this->output = '';
		$this->error = '';

		dol_syslog(__METHOD__, LOG_DEBUG);

		$now = dol_now();

		$this->db->begin();

		// ...

		$this->db->commit();

		return $error;
	}

	/**
	 * Save payment and invoice into middle table
	 *
	 * @param string $refpayment reference of the payment
	 * @param int $fk_facture id of the invoice
	 * @param float $amount amount of the payment
	 * @param string $tx transaction code
	 * @param string $code payment code
	 * @return int id of the inserted record or -1 if an error occurred
	 */
	public function savePayment($refpayment, $fk_facture, $amount, $tx, $code)
	{

		$sql = "INSERT INTO " . MAIN_DB_PREFIX . "cfdixml_payment_invoice (";
		$sql .= " ref_payment, fk_facture, amount, tx, code";
		$sql .= ") VALUES (";
		$sql .= "'" . $refpayment . "', " . $fk_facture . ", '" . $amount . "', '" . $tx . "', '" . $code . "'";
		$sql .= ")";

		dol_syslog("Save payment and invoice into middle table", LOG_DEBUG);
		$resql = $this->db->query($sql);
		if ($resql < 0) {
			dol_print_error($this->db);
			return -1;
		} else {
			$id = $this->db->last_insert_id(MAIN_DB_PREFIX . "cfdixml_payment_invoice");
			return $id;
		}
	}
	/**
	 * Delete a payment
	 *
	 * @param string $refpayment reference of the payment
	 *
	 * @return int 1 if success, -1 if error
	 */
	public function deletePayment($refpayment)
	{
		$sql = "DELETE FROM " . MAIN_DB_PREFIX . "cfdixml_payment_invoice WHERE ref_payment = '" . $refpayment . "'";
		$resql = $this->db->query($sql);
		if ($resql < 0) {
			dol_print_error($this->db);
			return -1;
		} else {
			return 1;
		}
	}



	public function printPaymentTable($object, $idpayment = 0)
	{
		$payments = $this->getPayments($object->fk_soc, $idpayment);
		$html = ''; // Variable para almacenar el código HTML de la tabla

		// Array para almacenar las facturas agrupadas por pay_ref
		$facturas_agrupadas = [];

		// Agrupar las facturas por pay_ref
		foreach ($payments as $objeto) {
			$pay_ref = $objeto->pay_ref;
			if (!isset($facturas_agrupadas[$pay_ref])) {
				$facturas_agrupadas[$pay_ref] = [];
			}
			$facturas_agrupadas[$pay_ref][] = $objeto;
		}

		// Inicio de la tabla HTML

		$html .= "<tr class=\"liste_titre\">";
		$html .= "<td class=\"liste_titre center\">Referencia de Pago</td>";
		$html .= "<td class=\"liste_titre center\">Fecha del pago</td>";
		$html .= "<td class=\"liste_titre center\">Referencia de Factura</td>";
		$html .= "<td class=\"liste_titre center\">Moneda de la factura</td>";
		$html .= "<td class=\"liste_titre center\">Total de Factura</td>";
		$html .= "<td class=\"liste_titre center\">Monto de Pago</td>";
		$html .= "<td class=\"liste_titre center\">Moneda Pago</td>";
		$html .= "<td class=\"liste_titre center\">Tipo de cambio</td>";
		$html .= "<td class=\"liste_titre center\">Total de Pago</td>";
		$html .= "<td class=\"liste_titre center\">Forma de Pago</td>";
		$html .= "<td class=\"liste_titre center\">Seleccionar</td>";
		$html .= "</tr>";
		// Recorrer las facturas agrupadas y generar filas de la tabla
		foreach ($facturas_agrupadas as $pay_ref => $facturas) {
			foreach ($facturas as $factura) {

				$html .= "<tr>";
				$html .= "<td class=\"center\">{$factura->pay_ref}</td>";
				$html .= "<td class=\"center\">" . dol_print_date($factura->date_payment, "%d/%m/%Y") . "</td>";
				$html .= "<td class=\"center\">{$factura->fact_ref}</td>";
				$html .= "<td class=\"center\">{$factura->code}</td>";
				$html .= "<td class=\"center\">" . price2num($factura->fact_total) . "</td>";
				$html .= "<td class=\"center\">" . price2num($factura->pay_amount) . "</td>";

				$code = $factura->payment_code ? $factura->payment_code : 'MXN';
				$html .= "<td class=\"center\">{$code}</td>";
				$html .= "<td class=\"center\">" . price2num($factura->payment_tx) . "</td>";
				$html .= "<td class=\"center\">" . price2num($factura->pay_total) . "</td>";
				$html .= "<td class=\"center\">{$object->getCodeFromId($factura->code_payment, 'c_paiement', 0, 1)}</td>";
				$html .= "<td class=\"center\"><input type='checkbox' name='pay_{$factura->pay_rowid}_{$factura->payfact_rowid}'  value='{$factura->pay_amount}'></td>";
				//$html .= "<input type='hidden' name='amount_{$factura->pay_rowid}'' />";
				$html .= "</tr>";
			}
		}

		$html .= "<tr><td class=\"center\" colspan=\"11\"><button type=\"submit\"><span class=\"fas fa-save\"></span>Guardar</button></td></tr>";
		return $html;
	}

	public function printPaymentLinesTable()
	{
		global $conf, $langs;

		$paiement = new Paiement($this->db);
		$facture = new Facture($this->db);

		$html = '';

		$lastPaymentId = null; // Variable para almacenar el último ID de pago procesado

		foreach ($this->lines as $line) {
			foreach ($line as $key => $value) {
				if ($key == 'fk_paiement') {
					$paiement->fetch($value);
					$sql = "SELECT ref_payment, fk_facture, amount, tx, code FROM " . MAIN_DB_PREFIX . "cfdixml_payment_invoice";
					$sql .= " WHERE ref_payment = '" . $paiement->ref . "'";
					$resql = $this->db->query($sql);
					$pay_arr = array();
					if ($resql) {
						$num = $this->db->num_rows($resql);
						if ($num > 0) {
							$i = 0;
							while ($i < $num) {
								$obj = $this->db->fetch_object($resql);
								$pay_arr[$obj->ref_payment] = [
									'fk_facture' => $obj->fk_facture,
									'amount' => $obj->amount,
									'tx' => $obj->tx,
									'code' => $obj->code
								];

								$i++;
							}
						}
					}

					// echo '<pre>';print_r($pay_arr);echo '</pre>';exit;
					// Verificar si el ID de pago es diferente al último procesado
					if ($paiement->id != $lastPaymentId) {
						// Nuevo pago, mostrar información de pago
						$html .= "<tr class=\"liste_titre\">";
						$html .= '<td colspan="2" class="linecoldescription">Pago</td>';
						$html .= '<td class="linecoldescription">Fecha</td>';
						$html .= '<td colspan="2" class="linecoldescription">Forma de Pago</td>';
						$html .= '<td colspan="2" class="linecoldescription">Importe</td>';
						$html .= '<td colspan="2" class="linecoldescription">Cuenta de Banco</td>';
						$html .= '<td colspan="2" class="linecoldescription">Entrada Bancaria</td>';
						$html .= '<td colspan="2" class="linecoldescription">Comentarios</td>';
						$html .= '<td class="linecoldescription">Moneda</td>';
						$html .= "</tr>";
						$html .= '<tr>';

						// Mostrar información del pago
						$html .= '<td colspan="2">' . $paiement->getNomUrl(1) . '</td>';
						$html .= '<td>' . dol_print_date($paiement->date, "%d/%m/%Y") . '</td>';
						$labeltype = $langs->trans("PaymentType" . $paiement->type_code) != ("PaymentType" . $paiement->type_code) ? $langs->trans("PaymentType" . $paiement->type_code) : $paiement->type_label;
						$html .= '<td colspan="2">' . $labeltype . '</td>';
						$html .= '<td colspan="2">' . price($paiement->multicurrency_amount, '', $langs, 0, -1, -1, $conf->currency) . '</td>';

						// Mostrar información de la cuenta bancaria
						if ($paiement->fk_account > 0) {
							$bankline = new AccountLine($this->db);
							$bankline->fetch($paiement->bank_line);
							$html .= '<td colspan="2">';
							$accountstatic = new Account($this->db);
							$accountstatic->fetch($bankline->fk_account);
							$html .= $accountstatic->getNomUrl(1);
							$html .= '</td>';
							$html .= '<td colspan="2">';
							$html .= $bankline->getNomUrl(1, 0, 'showconciliatedandaccounted');
							$html .= '</td>';
						} else {
							// Si no hay cuenta bancaria asociada
							$html .= '<td colspan="4">No hay cuenta bancaria asociada</td>';
						}

						$html .= '<td colspan="2">';
						$html .= $paiement->note_private;
						$html .= '</td>';
						$html .= '<td>';
						$html .= $pay_arr[$paiement->ref]['code'] ? $pay_arr[$paiement->ref]['code'] : $conf->currency;
						$html .= '</td>';
						$html .= '</tr>';
						$html .= '<tr>';
						$html .= '<td colspan="14">';
						$html .= '<table class="noborder noshadow" width="100%">';
						$html .= '<tr class="noborder" style="background-color:#ccc;">';
						$html .= '<td class="linecoldescription"></td>';
						$html .= '<td colspan="2" class="linecoldescription">Factura</td>';
						$html .= '<td class="linecoldescription">Total factura</td>';
						$html .= '<td class="linecoldescription">Saldo anterior</td>';
						$html .= '<td class="linecoldescription">Monto pago</td>';
						$html .= '<td class="linecoldescription">Parcialidad</td>';
						$html .= '<td class="linecoldescription">Saldo insoluto</td>';

						$html .= '</tr>';

						$rows = $this->listOfInvoices($paiement->id);

						foreach ($rows as $row => $line) {
							foreach ($line as $key => $value) {
								$facture->fetch($value->facid);

								$html .= '<tr>';
								$html .= '<td></td>';
								$html .= '<td colspan="2">' . $facture->getNomUrl(1) . '</td>';

								//REVISAR SI TAMBIÉN TOMA EN CUENTA SI ES MULTICURRENCY

								$html .= '<td>' . price($facture->multicurrency_total_ttc, 2, '', 1, 2, -1, $facture->multicurrency_code) . '</td>';

								// Aquí puedes realizar tus cálculos y mostrar los datos de saldo anterior, monto pago, etc.

								$num_fact = $this->getNumPaymentsFacture($paiement->id, $value->facid, $paiement->date);

								if (is_array($num_fact)) {
									$html .= '<td>' . price(($facture->multicurrency_total_ttc - $num_fact['amount']), 2, '', 1, 2, -1, $facture->multicurrency_code) . '</td>';
								} else {
									$html .= '<td>' . price($facture->multicurrency_total_ttc, 2, '', 1, 2, -1, $facture->multicurrency_code) . '</td>';
									$num_fact['nb'] = 1;
								}

								// $billsarray = $paiement->getBillsArray();
								$amounts_array = $paiement->getAmountsArray();

								if (isset($pay_arr[$paiement->ref]) && array_key_exists('code', $pay_arr[$paiement->ref])) {
									if ($pay_arr[$paiement->ref]['code'] != 'MXN') {
										if ($pay_arr[$paiement->ref]['fk_facture'] == $value->facid) {
											$html .= '<td>' . price($pay_arr[$paiement->ref]['amount'], 2, '', 1, 2, -1, $facture->multicurrency_code) . '</td>';
										}
									}
								} else {
									$html .= '<td>' . price($amounts_array[$value->facid], 2, '', 1, 2, -1, $facture->multicurrency_code) . '</td>';
								}


								$html .= '<td>' . $num_fact['nb'] . '</td>';
								$total_amounts_payed = $this->getPaymentsDateInvoice($value->facid, $paiement->date);
								if (isset($pay_arr[$paiement->ref]) && array_key_exists('code', $pay_arr[$paiement->ref])) {
									if ($pay_arr[$paiement->ref]['code'] != 'MXN') {
										if ($pay_arr[$paiement->ref]['fk_facture'] == $value->facid) {
											$diff = $facture->multicurrency_total_ttc - ($pay_arr[$paiement->ref]['amount']);
										}
									}
								} else {

									$diff = $facture->total_ttc - ($amounts_array[$value->facid] + $total_amounts_payed);
								}

								$html .= '<td>' . price($diff, 2, '', 1, 2, -1, $facture->multicurrency_code) . '</td>';

								$html .= '</tr>';
							}
						}

						$html .= '</table>';
						$html .= '</td>';
						$html .= '</tr>';

						// Actualizar el ID del último pago procesado
						$lastPaymentId = $paiement->id;
					}
				}
			}
		}

		return $html;
	}


	public function getNumPaymentsFacture($paiement_id, $facture_id, $fechaPago)
	{

		$result = array();

		$sql = "SELECT count(*) as nb, pf.*
        FROM " . MAIN_DB_PREFIX . "paiement_facture pf
        INNER JOIN " . MAIN_DB_PREFIX . "paiement p ON pf.fk_paiement = p.rowid
        WHERE pf.fk_facture = " . $facture_id . "
        AND pf.fk_paiement <> " . $paiement_id . "
        AND p.datep < '" . dol_print_date($fechaPago, '%Y-%m-%d') . "'";
		// echo $sql . '<br>';
		$resql = $this->db->query($sql);
		if ($resql) {
			$num = $this->db->num_rows($resql);
			if ($num > 0) {
				$i = 0;
				$sum = 0;
				while ($i < $num) {
					$obj = $this->db->fetch_object($resql);
					// echo '<pre>';print_r($obj->nb);echo '</pre>';
					if ((int)$obj->nb > 0) {
						$sum += price2num($obj->amount);
					}
					$i++;
				}
				$resullt['amount'] = $sum;
				$resullt['nb'] = $obj->nb + 1;
			}
		}
		return $resullt;

		$this->db->free($resql);
	}

	public function listOfInvoices($id)
	{
		/*
 		* List of invoices
 		*/

		$rows = array();
		$sql = 'SELECT f.rowid as facid, f.ref, f.type, f.total_ttc, f.paye, f.entity, f.fk_statut, pf.amount, s.nom as name, s.rowid as socid';
		$sql .= ' FROM ' . MAIN_DB_PREFIX . 'paiement_facture as pf,' . MAIN_DB_PREFIX . 'facture as f,' . MAIN_DB_PREFIX . 'societe as s';
		$sql .= ' WHERE pf.fk_facture = f.rowid';
		$sql .= ' AND f.fk_soc = s.rowid';
		$sql .= ' AND f.entity IN (' . getEntity('invoice') . ')';
		$sql .= ' AND pf.fk_paiement = ' . ((int) $id);
		$resql = $this->db->query($sql);
		if ($resql) {
			$num = $this->db->num_rows($resql);
			$i = 0;
			while ($i < $num) {
				$obj = $this->db->fetch_object($resql);
				$rows[$obj->facid][] = $obj;
				$i++;
			}
		}

		return $rows;
	}

	public function getPaymentsDateInvoice($id, $date)
	{
		$sql = 'SELECT p.datep as dp, p.ref, p.num_paiement as num_payment, p.rowid, p.fk_bank,';
		$sql .= ' c.code as payment_code, c.libelle as payment_label,';
		$sql .= ' pf.amount,';
		$sql .= ' ba.rowid as baid, ba.ref as baref, ba.label, ba.number as banumber, ba.account_number, ba.fk_accountancy_journal, ba.currency_code as bacurrency_code';
		$sql .= ' FROM ' . MAIN_DB_PREFIX . 'paiement_facture as pf, ' . MAIN_DB_PREFIX . 'paiement as p';
		$sql .= ' LEFT JOIN ' . MAIN_DB_PREFIX . 'c_paiement as c ON p.fk_paiement = c.id';
		$sql .= ' LEFT JOIN ' . MAIN_DB_PREFIX . 'bank as b ON p.fk_bank = b.rowid';
		$sql .= ' LEFT JOIN ' . MAIN_DB_PREFIX . 'bank_account as ba ON b.fk_account = ba.rowid';
		$sql .= ' WHERE pf.fk_facture = ' . ((int) $id) . ' AND pf.fk_paiement = p.rowid';
		$sql .= ' AND p.entity IN (' . getEntity('invoice') . ')';
		$sql .= ' AND p.datep < \'' . dol_print_date($date, '%Y-%m-%d') . '\''; // Fecha específica aquí, en formato 'YYYY-MM-DD'
		$sql .= ' ORDER BY p.datep, p.tms';
		// echo $sql.'<br>';
		$resql = $this->db->query($sql);
		if ($resql) {
			$num = $this->db->num_rows($resql);
			$i = 0;
			$suma = 0;
			while ($i < $num) {
				$obj = $this->db->fetch_object($resql);
				// $rows[$obj->ref][] = $obj;
				$suma = $suma + price2num($obj->amount);
				$i++;
			}
		}
		return $suma;
	}

	public function getCurrencyPayment($ref)
	{

		global $conf;
		$sql = " SELECT code from " . MAIN_DB_PREFIX . "cfdixml_payment_invoice where code = '" . $ref . "'";
		$resql = $this->db->query($sql);
		if ($resql) {
			$num = $this->db->num_rows($resql);
			if ($num > 0) {
				$obj = $this->db->fetch_object($resql);
				return $obj->code;
			} else {
				return $conf->currency;
			}
		}
	}
}


require_once DOL_DOCUMENT_ROOT . '/core/class/commonobjectline.class.php';

/**
 * Class PaymentLine. You can also remove this and generate a CRUD class for lines objects.
 */
class PaymentLine extends CommonObjectLine
{
	// To complete with content of an object PaymentLine
	// We should have a field rowid, fk_payment and position

	/**
	 * @var int  Does object support extrafields ? 0=No, 1=Yes
	 */
	public $isextrafieldmanaged = 0;

	/**
	 * Constructor
	 *
	 * @param DoliDb $db Database handler
	 */
	public function __construct(DoliDB $db)
	{
		$this->db = $db;
	}
	public $fields = array(
		'rowid' => array('type' => 'integer', 'label' => 'TechnicalID', 'enabled' => '1', 'position' => 1, 'notnull' => 1, 'visible' => 0, 'noteditable' => '1', 'index' => 1, 'css' => 'left', 'comment' => "Id"),
		'fk_payment' => array('type' => 'integer', 'label' => 'Payment ID', 'enabled' => '1', 'position' => 2, 'notnull' => 1, 'visible' => 1, 'comment' => "Payment ID"),
		'fk_paiement' => array('type' => 'integer', 'label' => 'Payment Dolibarr', 'enabled' => '1', 'position' => 2, 'notnull' => 1, 'visible' => 1, 'comment' => "Payment ID"),
		'fk_paiement_facture' => array('type' => 'integer', 'label' => 'Payment Invoice ID', 'enabled' => '1', 'position' => 2, 'notnull' => 1, 'visible' => 1, 'comment' => "Payment ID"),
		'amount' => array('type' => 'varchar(255)', 'label' => 'Amount', 'enabled' => '1', 'position' => 3, 'notnull' => 1, 'visible' => 1, 'comment' => "Amount"),
		'date_creation' => array('type' => 'datetime', 'label' => 'Date Created', 'enabled' => '1', 'position' => 4, 'notnull' => 1, 'visible' => 1, 'comment' => "Date Created"),
		'fk_user_creat' => array('type' => 'integer', 'label' => 'Created By', 'enabled' => '1', 'position' => 5, 'notnull' => 1, 'visible' => 1, 'comment' => "Created By"),
		'tms' => array('type' => 'timestamp', 'label' => 'Last Modified', 'enabled' => '1', 'position' => 6, 'notnull' => 1, 'visible' => 1, 'comment' => "Last Modified"),
	);

	public $rowid;
	public $fk_payment;
	public $fk_paiement_facture;
	public $fk_paiement;
	public $amount;
	public $date_creation;
	public $fk_user_creat;
	public $tms;
	public $lines = array();

	/**
	 * Add a new line to the payment
	 *
	 * @param User $user User that creates the line
	 * @param bool $notrigger false if we should trigger triggers after creating the line, true if not
	 * @return int 0 if KO, 1 if OK
	 */
	public function addLine(User $user, $notrigger = false)
	{
		global $langs;

		$now = dol_now();
		// Build SQL statement
		$sql = "INSERT INTO " . MAIN_DB_PREFIX . "cfdixml_payment_lines (";
		$sql .= "fk_payment,fk_paiement , fk_paiement_facture ,amount, date_creation, fk_user_creat";
		$sql .= ") VALUES (";
		$sql .= $this->fk_payment . ", " . $this->fk_paiement . ", " . $this->fk_paiement_facture . ", '" . price2num($this->amount) . "', '" . dol_print_date($now, "%Y-%m-%d") . "', " . $user->id . ")";
		// echo $sql;exit;
		dol_syslog(get_class($this) . "::addLine", LOG_DEBUG);

		$resql = $this->db->query($sql);
		if ($resql < 0) {
			dol_print_error($this->db);
			return -1;
		} else {
			$id = $this->db->last_insert_id(MAIN_DB_PREFIX . "cfdixml_payment_lines");
			return $id;
		}
	}

	public function fetchLines($id)
	{

		$sql = "SELECT ";
		foreach ($this->fields as $key => $value) {
			if ($key == 'rowid') {
				$sql .= $key . " as id,";
			} else {
				$sql .= $key . ",";
			}
		}
		$sql = substr($sql, 0, -1);
		$sql .= " FROM " . MAIN_DB_PREFIX . "cfdixml_payment_lines";
		$sql .= " WHERE fk_payment = " . $id;
		$sql .= " ORDER BY date_creation,fk_paiement DESC";

		dol_syslog(get_class($this) . "::fetchLines", LOG_DEBUG);
		$resql = $this->db->query($sql);
		if ($resql < 0) {
			dol_print_error($this->db);
			return -1;
		} else {
			$num = $this->db->num_rows($resql);
			if ($num) {
				$i = 0;
				while ($i < $num) {

					$obj = $this->db->fetch_object($resql);
					$this->lines[$i] = $obj;
					$i++;
				}
			}
			return  $this->lines;
			$this->db->free($resql);
		}
	}
}
