<?php
require_once DOL_DOCUMENT_ROOT . '/compta/facture/class/facture.class.php';
require_once DOL_DOCUMENT_ROOT . '/societe/class/societe.class.php';
require_once DOL_DOCUMENT_ROOT . '/product/class/product.class.php';
dol_include_once('/cfdixml/class/cfdi/CFDIManager.php');
dol_include_once('/cfdixml/class/cfdi/Certificado.php');
dol_include_once('/cfdixml/class/cfdi/CadenaOriginalBuilder.php');
dol_include_once('/cfdixml/class/cfdi/CFDIParametrosValidator.php');
dol_include_once('/cfdixml/class/cfdi/XSLTParser.php');
dol_include_once('/cfdixml/class/cfdi/complemento/AbstractComplementoManager.php');
dol_include_once('/cfdixml/class/cfdi/complemento/ComercioExteriorManager.php');
dol_include_once('/cfdixml/class/cfdifacturecce.class.php');

class InvoiceHandler
{
	private $db;
	private $user;
	private $langs;
	private $cfdiManager;
	private $validator;
	private $certificado;

	public function __construct($db, $user, $langs)
	{
		global $conf;
		$this->db = $db;
		$this->user = $user;
		$this->langs = $langs;

		$xsltParser = new XSLTParser($conf->global->CFDIXML_XSLT_PATH);
		$this->validator = new CFDIParametrosValidator($xsltParser->generarEstructuraParametros());
		$this->cfdiManager = new CFDIManager($conf->global->CFDIXML_XSLT_PATH);
		$this->certificado = new Certificado();
	}

	/**
	 * Handles the stamping process for an invoice.
	 *
	 * @param Facture $object The invoice object to be stamped
	 * @param array $parameters Required parameters
	 * @param bool $debug Whether to show debug information
	 * @return int 1 if success, 0 or negative if error
	 */
	public function handleStamping(&$object, $parameters, $debug = false)
	{
		global $conf;

		if (!$this->canBeStamped($object)) {
			return $this->handleError('FacturaNoPuedeTimbrarse');
		}

		if (!$this->validateRequiredData($parameters)) {
			return $this->handleError('DatosRequeridosFaltantes');
		}

		try {
			$result = $this->processStamping($object, $parameters, $debug);
			return $result > 0 ? 1 : 0;
		} catch (Exception $e) {
			return $this->handleError($e->getMessage());
		}
	}

	/**
	 * Determines if an invoice can be stamped
	 */
	private function canBeStamped($object)
	{
		return (in_array($object->status, [1, 4]) &&
				empty($object->array_options["options_cfdixml_UUID"])) ||
			($object->module_source &&
				empty($object->array_options["options_cfdixml_UUID"])) ||
			($object->pos_source > 0 &&
				empty($object->array_options["options_cfdixml_UUID"]));
	}

	/**
	 * Validates required data for stamping
	 */
	private function validateRequiredData($parameters)
	{
		$required = [
			'uso_cfdi',
			'condicion_pago',
			'forma_pago',
			'metodo_pago',
			'exportacion'
		];

		foreach ($required as $field) {
			if ($parameters[$field] < 0) return false;
		}
		return true;
	}

	/**
	 * Processes the stamping of an invoice.
	 */
	private function processStamping(&$object, $parameters, $debug = false)
	{
		global $conf;

		try {
			// Validar datos iniciales
			$stampingData = $this->validateAndGetStampingData($parameters);
			if (!$stampingData) {
				return -1;
			}

			// Actualizar factura con datos preliminares
			if (!$this->updateInvoicePreliminaryData($object, $stampingData)) {
				return -1;
			}

			// Preparar datos para el CFDI
			$cfdiData = $this->prepareCFDIData($object, $stampingData);

			// Generar y validar CFDI
			try {
				$xml = $this->cfdiManager->generarCFDI($cfdiData, $debug);
			} catch (Exception $e) {
				return $this->handleError('Error al generar CFDI: ' . $e->getMessage());
			}

			// Guardar XML preliminar
			if (!$this->savePreliminaryXml($object, $xml)) {
				return -1;
			}

			// Timbrar con PAC
			$stampResult = $this->stampWithPAC($object, $xml, $debug);
			if (!$stampResult) {
				return -1;
			}

			// Procesar respuesta y actualizar factura
			if (!$this->processPACResponse($object, $stampResult)) {
				return -1;
			}

			// Manejar documentos relacionados
			if ($object->type == Facture::TYPE_REPLACEMENT) {
				if (!$this->handleRelatedDocuments($object, $stampResult)) {
					return -1;
				}
			}

			return $this->finishStampingProcess($object);

		} catch (Exception $e) {
			dol_syslog("Error en processStamping: " . $e->getMessage(), LOG_ERR);
			return $this->handleError('Error en el proceso de timbrado: ' . $e->getMessage());
		}
	}

	/**
	 * Handles the stamping with PAC
	 */
	private function stampWithPAC(&$object, $xml, $debug = false)
	{
		global $conf;
		try {
			$preparedXML = $this->prepareXMLForStamping($xml);

			$params = array(
				"xml" => $preparedXML,
				"username" => strtolower($conf->global->MAIN_INFO_SIREN),
				"password" => $conf->global->CFDIXML_WS_TOKEN
			);

			if ($debug) {
				echo "<pre>Parámetros para el PAC:\n";
				print_r($params);
				echo "</pre>";
			}

			$client = new SoapClient($conf->global->CFDIXML_WS_URL . '/servicios/soap/stamp.wsdl');
			echo $conf->global->CFDIXML_WS_URL;
			$response = $client->__soapCall("stamp", array($params));
			$debug = true;
			if ($debug) {
				echo "<pre>Respuesta del PAC:\n";
				print_r($response);
				echo "</pre>";
			}

			// Verificar respuesta válida
			if (!isset($response->stampResult)) {
				throw new Exception("Respuesta del PAC inválida");
			}

			// Procesar incidencias si existen
			if (isset($response->stampResult->Incidencias) && !empty((array)$response->stampResult->Incidencias)) {
				$incidencia = $response->stampResult->Incidencias->Incidencia;

				if ($incidencia->CodigoError == '307') {
					if ($debug) {
						echo "<pre>Detectado timbre previo. Intentando recuperar...\n</pre>";
					}
					return $this->handlePreviousStamp($client, $params);
				}

				throw new Exception(sprintf(
					'Error %s: %s (WorkProcessId: %s)',
					$incidencia->CodigoError,
					$incidencia->MensajeIncidencia,
					$incidencia->WorkProcessId
				));
			}

			// Procesar respuesta exitosa
			if (!empty($response->stampResult->UUID)) {
				return [
					'code' => '200',
					'data' => $response->stampResult->xml,
					'uuid' => $response->stampResult->UUID,
					'fecha' => $response->stampResult->Fecha,
					'selloSAT' => $response->stampResult->SatSeal,
					'certificadoSAT' => $response->stampResult->NoCertificadoSAT
				];
			}

			throw new Exception("La respuesta del PAC no contiene un UUID válido");

		} catch (Exception $e) {
			setEventMessages($e->getMessage(), null, 'errors');
			dol_syslog($e->getMessage(), LOG_ERR);
			return false;
		}
	}

	/**
	 * Prepara el XML para timbrado
	 */
	private function prepareXMLForStamping($xml)
	{
		// 1. Verificar que el XML es válido
		$doc = new DOMDocument();
		$doc->loadXML($xml);

		// 2. Verificar namespaces y estructura básica
		$rootElement = $doc->documentElement;
		if ($rootElement->nodeName !== 'cfdi:Comprobante') {
			throw new Exception('El elemento raíz debe ser cfdi:Comprobante');
		}

		// 3. Verificar atributos requeridos
		$requiredAttrs = [
			'Version',
			'Fecha',
			'Sello',
			'NoCertificado',
			'Certificado',
			'SubTotal',
			'Moneda',
			'Total',
			'TipoDeComprobante',
			'Exportacion',
			'LugarExpedicion'
		];

		foreach ($requiredAttrs as $attr) {
			if (!$rootElement->hasAttribute($attr)) {
				throw new Exception("Falta el atributo requerido: $attr");
			}
		}

		// 4. Formatear el XML correctamente
		$doc->formatOutput = true;
		$doc->preserveWhiteSpace = false;
		$formattedXML = $doc->saveXML();

		// 5. Verificar codificación
		if (!mb_check_encoding($formattedXML, 'UTF-8')) {
			$formattedXML = mb_convert_encoding($formattedXML, 'UTF-8');
		}

		return $formattedXML;
	}

	private function prepareCFDIData(&$object, $data)
	{
		global $conf;

		try {
			// 1. Verificar y procesar certificado
			$certificadoInfo = $this->processCertificate($conf->global->CFDIXML_CER_FILE);

			// 2. Preparar datos del receptor
			$societe = new Societe($this->db);
			$societe->fetch($object->socid);

			// 3. Generar datos base del CFDI
			$cfdiData = $this->prepareCFDIBaseData($object, $data, $certificadoInfo, $societe);
			// 4. Verificar si hay información de Comercio Exterior
			$cfdiData['Complementos'] = $this->prepareComplementos($object, $data);

			return $cfdiData;

		} catch (Exception $e) {
			throw new Exception("Error preparando datos CFDI: " . $e->getMessage());
		}
	}

	private function processCertificate($certificatePath)
	{
		$verification = $this->certificado->verificarCertificado($certificatePath);
		if (!$verification['valido']) {
			throw new Exception("Certificado inválido: " . $verification['mensaje']);
		}
		return $this->certificado->procesarCertificado($certificatePath);
	}

	private function prepareCFDIBaseData($object, $data, $certificadoInfo, $societe)
	{
		global $conf;

		// Preparar serie y folio
		$serieFolio = explode('-', $object->ref);
		$serie = $serieFolio[0] ?? '';
		$folio = $serieFolio[1] ?? $serieFolio[0];

		// Preparar conceptos
		$conceptos = $this->getConceptosData($object);

		if (in_array($object->type, [Facture::TYPE_STANDARD, Facture::TYPE_DEPOSIT, Facture::TYPE_REPLACEMENT])) {
			$TipoDeComprobante = 'I';
		} else if ($object->type == Facture::TYPE_CREDIT_NOTE) {
			$TipoDeComprobante = 'E';
		}
		// Retornar datos base del CFDI
		return [
			'Comprobante' => [
				'Version' => '4.0',
				'Serie' => $serie,
				'Folio' => $folio,
				'Fecha' => $object->array_options['options_cfdixml_control'],
				'NoCertificado' => $certificadoInfo['noCertificado'],
				'Certificado' => $certificadoInfo['certificado'],
				'FormaPago' => $data['forma_pago'],
				'CondicionesDePago' => $object->cond_reglement_doc,
				'MetodoPago' => $data['metodo_pago'],
				'TipoDeComprobante' => $TipoDeComprobante,
				'LugarExpedicion' => $conf->global->MAIN_INFO_SOCIETE_ZIP,
				'Exportacion' => $data['exportacion'],
				'Moneda' => $object->multicurrency_code ?: 'MXN',
				'TipoCambio' => $this->getTipoCambio($object, $data),
				'SubTotal' => number_format($object->total_ht, 2, '.', ''),
				'Total' => number_format($object->total_ttc, 2, '.', '')
			],
			'Emisor' => $this->getEmisorData(),
			'Receptor' => $this->getReceptorData($object, $societe),
			'Conceptos' => $conceptos,
			'Certificado' => [
				'certificadoPath' => $conf->global->CFDIXML_CER_FILE,
				'llavePrivadaPath' => $conf->global->CFDIXML_KEY_FILE,
				'password' => $conf->global->CFDIXML_CERKEY_PASS
			]
		];


	}

	private function getTipoCambio($object, $data)
	{
		if ($object->multicurrency_code !== 'MXN') {
			return $data['tipo_cambio_usd'] ?? $object->multicurrency_tx;
		}
		return "1";
	}

	private function prepareComplementos($object, $data)
	{
		$complementos = [];

		// Agregar Complemento Comercio Exterior si aplica
		if ($data['exportacion'] === '02') {
			$complementos['ComercioExterior'] = $this->getComercioExteriorData($object, $data);
		}

		return $complementos;
	}

	private function getComercioExteriorData($object, $data)
	{
		global $conf, $mysoc;
		$sqlitePath = dol_buildpath('/cfdixml/catalogs/catalogos.sqlite3', 0);

//		echo '<pre>';print_r($mysoc);exit;
		$objectCce = new CfdiFactureCce($this->db);
		if ($objectCce->fetch(0, $object->id) > 0) {
			// Obtener datos del receptor
			$societe = new Societe($this->db);
			$societe->fetch($object->socid);
			$municipio = $this->getCatalogFromSQLite($sqlitePath, 'cce_20_municipios', 'texto', $mysoc->town, true, 'municipio');
			$localidad = $this->getCatalogFromSQLite($sqlitePath, 'cce_20_localidades', 'texto', $conf->global->CFDIXML_LOCALIDAD, true, 'localidad');
			$colonia = $this->getCatalogFromSQLite($sqlitePath, 'cce_20_colonias', 'codigo_postal', $mysoc->zip, true, 'colonia');

//			echo $localidad;exit;
			// Estructura del complemento Comercio Exterior
//			echo  $mysoc->state_code;exit;
			$data = [
				'Version' => $objectCce->version,
				'ClaveDePedimento' => $objectCce->clave_pedimento,
				'CertificadoOrigen' => $objectCce->certificado_origen ? $objectCce->certificado_origen : '0',
				'Incoterm' => $objectCce->incoterm,
				'TipoCambioUSD' => number_format($objectCce->tipo_cambio_usd, 6, '.', ''),
				'TotalUSD' => number_format($objectCce->total_usd, 2, '.', ''),
				'Emisor' => [
					'Curp' => 'XIQB891116MGRMZR05', //borrar solo pruebas y hay que poner validación para
					'Domicilio' => [
						'Calle' => $mysoc->address ?? 'SIN CALLE',
						'NumeroExterior' => $conf->global->CFDIXML_NUMERO_EXTERIOR ?? 'N/A',
						'NumeroInterior' => $conf->global->CFDIXML_NUMERO_INTERIOR ?? '',
						'Colonia' => $conf->global->CFDIXML_COLONIA ?? '',
						'Localidad' => $conf->global->CFDIXML_LOCALIDAD ?? '',
						'Municipio' => $conf->global->CFDIXML_MUNICIPIO ?? '',
						'Estado' => $conf->global->CFDIXML_ESTADO ?? '',
						'Pais' => 'MEX',
						'CodigoPostal' => $mysoc->zip ?? ''
					]
				],
				'Receptor' => [
					'NumRegIdTrib' => $societe->idprof1 ?? 'XEXX010101000',
					'Domicilio' => [
						'Calle' => $societe->address ?? 'SIN CALLE',
						'NumeroExterior' => $societe->array_options['options_numero_exteriorcce'] ?? 'N/A',
						'Localidad' => $societe->town ?? '',
						'Municipio' => $societe->array_options['options_municipiocce'] ?? '',
						'Estado' => $societe->state_code ?? '',
						'Pais' => $this->getCountryCode($societe->country_code) ?? 'MEX',
						'CodigoPostal' => $societe->zip ?? ''
					]
				],
				'Mercancias' => $this->getMercanciasData($object)
			];

			if (!empty($objectCce->numero_exportador_confiable)) {
				$data['NumExportadorConfiable'] = $objectCce->numero_exportador_confiable;
			}
			if(!empty($objectCce->observaciones)){
				$data['Observaciones'] = $objectCce->observaciones;
			}
			if(!empty($objectCce->num_certificado_origen)){
				$data['NumCertificadoOrigen'] = $objectCce->num_certificado_origen;
			}
			if(!empty($objectCce->motivo_traslado) && (int)$objectCce->motivo_traslado > 0 ){

				$data['MotivoTraslado'] = $objectCce->motivo_traslado;
			}
			return $data;
		}

		throw new Exception("No se encontró información de Comercio Exterior para esta factura.");
	}

	private function getConceptosData($object)
	{
		$conceptos = [];
		foreach ($object->lines as $line) {
			if ($line->fk_product > 0) {  // Si es un producto
				$product = new Product($this->db);
				$product->fetch($line->fk_product);


				$concepto = [
					'ClaveProdServ' => $product->array_options['options_cfdixml_claveprodserv'],
					'NoIdentificacion' => $product->ref,
					'Cantidad' => $line->qty,
					'ClaveUnidad' => $product->array_options['options_cfdixml_umed'],
					'Descripcion' => $product->label,
					'ValorUnitario' => number_format($line->subprice, 2, '.', ''),
					'Importe' => number_format($line->total_ht, 2, '.', ''),
					'ObjetoImp' => $product->array_options['options_cfdixml_objetoimp'] ?? '02'
				];
			} else {  // Si es una línea libre
				$concepto = [
					'ClaveProdServ' => $line->array_options['options_cfdixml_claveprodserv'],
					'NoIdentificacion' => $line->ref ?? 'SERVICIO',
					'Cantidad' => $line->qty,
					'ClaveUnidad' => $line->array_options['options_cfdixml_umed'],
					'Descripcion' => $line->desc ?? $line->label,
					'ValorUnitario' => number_format($line->subprice, 2, '.', ''),
					'Importe' => number_format($line->total_ht, 2, '.', ''),
					'ObjetoImp' => $line->array_options['options_cfdixml_objetoimp'] ?? '02'
				];
			}

			// Agregar descuento si existe
			if ($line->remise_percent > 0) {
				$descuento = $line->total_ht * ($line->remise_percent / 100);
				$concepto['Descuento'] = number_format($descuento, 2, '.', '');
			}

			// Agregar impuestos si aplica
			if ($concepto['ObjetoImp'] === '02') {
				if ($line->tva_tx > 0) {
					$concepto['Impuestos']['Traslados'][] = [
						'Base' => number_format($line->total_ht, 2, '.', ''),
						'Impuesto' => '002',
						'TipoFactor' => 'Tasa',
						'TasaOCuota' => number_format($line->tva_tx / 100, 6, '.', ''),
						'Importe' => number_format($line->total_tva, 2, '.', '')
					];

				} else {
					// IVA 0%
					$concepto['Impuestos']['Traslados'][] = [
						'Base' => number_format($line->total_ht, 2, '.', ''),
						'Impuesto' => '002',
						'TipoFactor' => 'Tasa',
						'TasaOCuota' => '0.000000',
						'Importe' => '0.00'
					];

					if($object->array_options['options_cfdixml_exportacion'] == '02'){
						$concepto['ObjetoImp'] = '01';
						unset($concepto['Impuestos']);
					}
				}

				if((float)$line->localtax1_tx > 0){
					$concepto['Impuestos']['Traslados'][] = [
                        'Base' => number_format($line->total_ht, 2, '.', ''),
                        'Impuesto' => '003',
                        'TipoFactor' => 'Tasa',
                        'TasaOCuota' => number_format((float)$line->localtax1_tx / 100, 6, '.', ''),
                        'Importe' => number_format($line->total_localtax1, 2, '.', '')
                    ];
				}
			}

			$conceptos[] = $concepto;
		}
		return $conceptos;
	}

	private function getMercanciasData($object)
	{
		$mercancias = [];
		$objectCce = new CfdiFactureCce($this->db);
		$objectCce->fetch(0,$object->id);

		foreach ($object->lines as $line) {
			$valorUnitarioAduana = (float)$line->subprice / (float)$objectCce->tipo_cambio_usd;
			$valorDolares = (float)$line->total_ht / (float)$objectCce->tipo_cambio_usd;
			if ($line->fk_product > 0) {
				$product = new Product($this->db);
				$product->fetch($line->fk_product);
				$mercancias[] = [
					'NoIdentificacion' => $product->ref,
					'FraccionArancelaria' => $product->array_options['options_cfdixml_fraccionarancelariacce'] ?? '',
					'CantidadAduana' => $line->qty,
					'UnidadAduana' => $product->array_options['options_cfdixml_unidadaduanacce'] ?? '',
					'ValorUnitarioAduana' => number_format($valorUnitarioAduana, 2, '.', ''),
					'ValorDolares' => number_format($valorDolares, 2, '.', '')
				];
			} else {
				$mercancias[] = [
					'NoIdentificacion' => $line->ref,
					'FraccionArancelaria' => $line->array_options['options_cfdixml_fraccionarancelariacce'] ?? '',
					'CantidadAduana' => $line->qty,
					'UnidadAduana' => $line->array_options['options_cfdixml_unidadaduanacce'] ?? '',
					'ValorUnitarioAduana' => number_format($valorUnitarioAduana, 2, '.', ''),
					'ValorDolares' => number_format($valorDolares, 2, '.', '')
				];
			}
		}
		return $mercancias;
	}

//	private function prepareCFDIData(&$object, $data) {
//		global $conf;
//
//		try {
//			// Verificar certificado primero
//			$verificacion = $this->certificado->verificarCertificado($conf->global->CFDIXML_CER_FILE);
//			if (!$verificacion['valido']) {
//				throw new Exception("Certificado inválido: " . $verificacion['mensaje']);
//			}
//
//			// Procesar certificado
//			$certificadoInfo = $this->certificado->procesarCertificado($conf->global->CFDIXML_CER_FILE);
//
//			$societe = new Societe($this->db);
//			$societe->fetch($object->socid);
//
//			// Preparar serie y folio
//			$serie_folio = explode('-', $object->ref);
//			$serie = (count($serie_folio) > 1) ? $serie_folio[0] : '';
//			$folio = (count($serie_folio) > 1) ? $serie_folio[1] : $serie_folio[0];
//
//			// Manejar tipo de cambio
//			$tipoCambio = ($object->multicurrency_code != 'MXN') ? $object->multicurrency_tx : "1";
//
//			// Preparar conceptos con impuestos
//			$conceptos = [];
//			foreach ($object->lines as $line) {
//				$concepto = [];
//				$importe = $line->total_ht;
//				$valorUnitario = $line->subprice;
//
//				// Determinar valores basados en si existe producto o no
//				$claveProdServ = '';
//				$claveUnidad = '';
//				$nombreUnidad = '';
//				$objetoImp = '02';
//
//				if ($line->fk_product > 0) {
//					$product = new Product($this->db);
//					$product->fetch($line->fk_product);
//
//					// Obtener valores del producto
//					$claveProdServ = $product->array_options['options_cfdixml_claveprodserv'];
//					$claveUnidad = $product->array_options['options_cfdixml_umed'];
//					$nombreUnidad = $product->array_options['options_cfdixml_nombreumed'];
//					$objetoImp = $product->array_options['options_cfdixml_objetoimp'] ?? '02';
//				} else {
//					// Obtener valores de la línea
//					$claveProdServ = $line->array_options['options_cfdixml_claveprodserv'];
//					$claveUnidad = $line->array_options['options_cfdixml_umed'];
//					$nombreUnidad = $line->array_options['options_cfdixml_nombreumed'];
//					$objetoImp = $line->array_options['options_cfdixml_objetoimp'] ?? '02';
//				}
//
//				$concepto = [
//					'ClaveProdServ' => $claveProdServ,
//					'NoIdentificacion' => $line->ref,
//					'Cantidad' => $line->qty,
//					'ClaveUnidad' => $claveUnidad,
//					'Unidad' => $nombreUnidad,
//					'Descripcion' => $line->desc ?: $line->product_label,
//					'ValorUnitario' => number_format($valorUnitario, 2, '.', ''),
//					'Importe' => number_format($importe, 2, '.', ''),
//					'ObjetoImp' => $objetoImp
//				];
//				if($object->cce){
//					$objetoImp = '01';
//
//					$concepto['FraccionArancelaria'] = $product->array_options['options_cfdixml_fraccionarancelariacce']?? '';
//					$concepto['UnidadAduana'] = $product->array_options['options_cfdixml_unidadaduanacce']?? '';
//				}
//				if ($objetoImp === '02') {
//					$concepto['Impuestos'] = [
//						'Retenciones' => [],
//						'Traslados' => []
//					];
//
//					// Agregar retenciones ISR si aplica
//					if ($line->total_localtax1 < 0) {
//						$concepto['Impuestos']['Retenciones'][] = [
//							'Base' => number_format($importe, 2, '.', ''),
//							'Impuesto' => '001',
//							'TipoFactor' => 'Tasa',
//							'TasaOCuota' => '0.012500',
//							'Importe' => number_format(abs($line->total_localtax1), 2, '.', '')
//						];
//					}
//
//					// Agregar traslados IVA
//					if ($line->tva_tx > 0) {
//						$concepto['Impuestos']['Traslados'][] = [
//							'Base' => number_format($importe, 2, '.', ''),
//							'Impuesto' => '002',
//							'TipoFactor' => 'Tasa',
//							'TasaOCuota' => number_format($line->tva_tx/100, 6, '.', ''),
//							'Importe' => number_format($line->total_tva, 2, '.', '')
//						];
//					}
//				}
//
//				// Agregar descuento si existe
//				if ($line->remise_percent > 0) {
//					$descuento = $importe * ($line->remise_percent / 100);
//					$concepto['Descuento'] = number_format($descuento, 2, '.', '');
//				}
//
//				$conceptos[] = $concepto;
//			}
//
//			return [
//				'Comprobante' => [
//					'Version' => '4.0',
//					'Serie' => $serie,
//					'Folio' => $folio,
//					'Fecha' => $object->array_options['options_cfdixml_control'],
//					'NoCertificado' => $certificadoInfo['noCertificado'],
//					'Certificado' => $certificadoInfo['certificado'],
//					'FormaPago' => $data['forma_pago'],
//					'CondicionesDePago' => $object->cond_reglement_doc,
//					'MetodoPago' => $data['metodo_pago'],
//					'TipoDeComprobante' => 'I',
//					'LugarExpedicion' => $conf->global->MAIN_INFO_SOCIETE_ZIP,
//					'Exportacion' => $data['exportacion'],
//					'Moneda' => $object->multicurrency_code ?: 'MXN',
//					'TipoCambio' => $tipoCambio,
//					'SubTotal' => number_format($object->total_ht, 2, '.', ''),
//					'Total' => number_format($object->total_ttc, 2, '.', '')
//				],
//				'Emisor' => $this->getEmisorData(),
//				'Receptor' => $this->getReceptorData($object, $societe),
//				'Conceptos' => $conceptos,
//				'Certificado' => [
//					'certificadoPath' => $conf->global->CFDIXML_CER_FILE,
//					'llavePrivadaPath' => $conf->global->CFDIXML_KEY_FILE,
//					'password' => $conf->global->CFDIXML_CERKEY_PASS
//				]
//			];
//
//		} catch (Exception $e) {
//			throw new Exception("Error preparando datos CFDI: " . $e->getMessage());
//		}
//	}

	public function getEmisorData()
	{
		global $conf;
		$data = [
			'Rfc' => $conf->global->MAIN_INFO_SIREN,
			'Nombre' => strtoupper($conf->global->MAIN_INFO_SOCIETE_NOM),
			'RegimenFiscal' => getRegimenFiscal($conf->global->MAIN_INFO_SOCIETE_FORME_JURIDIQUE)
		];

		return $data;
	}

	public function getReceptorData(Facture $object, Societe $societe)
	{

		global $db, $langs, $conf;
		$data = [];

		if ($societe->idprof1 == "XAXX010101000") {
			$data = [
				'Rfc' => "XAXX010101000",
				'Nombre' => strtoupper('PUBLICO EN GENERAL'),
				'DomicilioFiscalReceptor' => $conf->global->MAIN_INFO_SOCIETE_ZIP,
				'RegimenFiscalReceptor' => '616',
				'UsoCFDI' => $object->array_options['options_cfdixml_usocfdi']
			];

			return $data;
		}

		if ($societe->country_code != 'MX') {

			$sql = "SELECT code_iso FROM " . MAIN_DB_PREFIX . "c_country WHERE code = '" . $societe->country_code . "' LIMIT 1";
			$result = $db->query($sql);
			$obj = $db->fetch_object($result);
			$data = [
				'Rfc' => "XEXX010101000",
				'Nombre' => strtoupper($societe->name),
				'DomicilioFiscalReceptor' => $conf->global->MAIN_INFO_SOCIETE_ZIP,
				'RegimenFiscalReceptor' => '616',
				'UsoCFDI' => $object->array_options['options_cfdixml_usocfdi']
			];
			if($object->array_options['options_cfdixml_exportacion'] == '02'){
				$data['ResidenciaFiscal'] = $this->getCountryCode($societe->country_code);
				$data['NumRegIdTrib'] = $societe->idprof1;
			}
			return $data;
		}

		$data = [

			'Rfc' => $societe->idprof1,
			'Nombre' => strtoupper($societe->name),
			'DomicilioFiscalReceptor' => $societe->zip,
			'RegimenFiscalReceptor' => getRegimenFiscal($societe->forme_juridique_code),
			'UsoCFDI' => $object->array_options['options_cfdixml_usocfdi']

		];

		return $data;
	}

	/**
	 * Maneja timbres previos
	 */
	private function handlePreviousStamp($client, $params)
	{
		try {
			// Llamar al método stamped del PAC
			$stamped = $client->__soapCall("stamped", array($params));

			if (!isset($stamped->stampedResult)) {
				throw new Exception('Respuesta inválida del PAC al recuperar timbre previo');
			}

			// Verificar si tenemos un UUID pero no XML
			if (!empty($stamped->stampedResult->UUID) && empty($stamped->stampedResult->xml)) {
				// Intentar recuperar el CFDI usando el UUID
				$recoveryParams = array(
					"uuid" => $stamped->stampedResult->UUID,
					"username" => $params['username'],
					"password" => $params['password']
				);

				// Llamar al método recover del PAC
				$recovered = $client->__soapCall("recover", array($recoveryParams));

				if (isset($recovered->recoverResult) && !empty($recovered->recoverResult->xml)) {
					return [
						'code' => '200',
						'data' => $recovered->recoverResult->xml,
						'uuid' => $stamped->stampedResult->UUID,
						'fecha' => $stamped->stampedResult->Fecha,
						'selloSAT' => $stamped->stampedResult->SatSeal,
						'certificadoSAT' => $stamped->stampedResult->NoCertificadoSAT
					];
				}
			}

			// Si tenemos el XML directamente del stamped
			if (!empty($stamped->stampedResult->xml)) {
				return [
					'code' => '200',
					'data' => $stamped->stampedResult->xml,
					'uuid' => $stamped->stampedResult->UUID,
					'fecha' => $stamped->stampedResult->Fecha,
					'selloSAT' => $stamped->stampedResult->SatSeal,
					'certificadoSAT' => $stamped->stampedResult->NoCertificadoSAT
				];
			}

			throw new Exception(sprintf(
				'XML no encontrado para UUID: %s (Fecha: %s)',
				$stamped->stampedResult->UUID,
				$stamped->stampedResult->Fecha
			));
		} catch (Exception $e) {
			throw new Exception('Error recuperando timbre previo: ' . $e->getMessage());
		}
	}

	/**
	 * Procesa la respuesta del PAC
	 */
	public function processPACResponse($object, $stampResult)
	{
		if (!isset($stampResult['data']) || !isset($stampResult['uuid'])) {
			return $this->handleError('Respuesta del PAC incompleta');
		}

		try {
			$this->db->begin();

			// Actualizar datos del CFDI
			$this->updateCFDIData($object, $stampResult);

			// Guardar XML timbrado
			$this->saveFinalXML($object, $stampResult);

			$this->db->commit();
			return true;
		} catch (Exception $e) {
			$this->db->rollback();
			return $this->handleError('Error procesando respuesta PAC: ' . $e->getMessage());
		}
	}

	/**
	 * Actualiza los datos del CFDI en la factura
	 */
	private function updateCFDIData(&$object, $stampResult)
	{
		if (empty($stampResult['data'])) {
			throw new Exception('XML vacío en la respuesta del PAC');
		}

		$facture = new Facture($this->db);
		$facture->fetch($object->id);

		$xml = new DOMDocument();
		$loadResult = @$xml->loadXML($stampResult['data']);

		if (!$loadResult) {
			throw new Exception('Error al cargar el XML: ' . libxml_get_last_error()->message);
		}

		// Obtener TFD namespace
		$tfd = $xml->getElementsByTagNameNS('http://www.sat.gob.mx/TimbreFiscalDigital', 'TimbreFiscalDigital')->item(0);

		if (!$tfd) {
			throw new Exception('No se encontró el nodo TimbreFiscalDigital en el XML');
		}

		$facture->array_options['options_cfdixml_UUID'] = $tfd->getAttribute('UUID');
		$facture->array_options['options_cfdixml_fechatimbrado'] = $tfd->getAttribute('FechaTimbrado');
		$facture->array_options['options_cfdixml_sellosat'] = $tfd->getAttribute('SelloSAT');
		$facture->array_options['options_cfdixml_certsat'] = $tfd->getAttribute('NoCertificadoSAT');
		$facture->array_options['options_cfdixml_sellocfd'] = $tfd->getAttribute('SelloCFD');
		$facture->array_options['options_cfdixml_certcfd'] = $xml->documentElement->getAttribute('NoCertificado');
		$facture->array_options['options_cfdixml_cadenaorig'] = $this->generateCadenaOriginal($stampResult['data']);
		$facture->array_options['options_cfdixml_xml'] = base64_encode($stampResult['data']);
		$facture->array_options['options_cfdixml_control'] = '';

		$this->db->begin();
		if ($facture->update($this->user, 1) <= 0) {
			$this->db->rollback();
			throw new Exception('Error actualizando datos del CFDI en la base de datos');
		}
		$this->db->commit();
	}

	/**
	 * Guarda el XML final
	 */
	private function saveFinalXML(&$object, $stampResult)
	{
		global $conf;

		$xml = new DOMDocument();
		$xml->loadXML($stampResult['data']);
		$tfd = $xml->getElementsByTagNameNS('http://www.sat.gob.mx/TimbreFiscalDigital', 'TimbreFiscalDigital')->item(0);
		$uuid = $tfd->getAttribute('UUID');

		$filedir = $conf->facture->multidir_output[$object->entity] . '/' .
			dol_sanitizeFileName($object->ref);

		$filename = $filedir . "/" . $object->ref . '_' . $uuid . ".xml";

		if (!file_put_contents($filename, $stampResult['data'])) {
			throw new Exception('Error al guardar el XML final');
		}
	}

	/**
	 * Guarda el XML preliminar
	 */
	private function savePreliminaryXml(&$object, $xml)
	{
		global $conf;

		$filedir = $conf->facture->multidir_output[$object->entity] . '/' .
			dol_sanitizeFileName($object->ref);

		if (!is_dir($filedir)) {
			mkdir($filedir, 0777, true);
		}

		try {
			$file_xml = fopen($filedir . "/PRECFDI_" . $object->ref . ".xml", "w");
			fwrite($file_xml, mb_convert_encoding($xml, 'utf8'));
			fclose($file_xml);
			return true;
		} catch (Exception $e) {
			return $this->handleError('Error al guardar XML: ' . $e->getMessage());
		}
	}

	/**
	 * Extrae el sello CFD del XML
	 */
	private function extractSelloCFD($xml)
	{
		$doc = new DOMDocument();
		$doc->loadXML($xml);
		$tfd = $doc->getElementsByTagNameNS('http://www.sat.gob.mx/TimbreFiscalDigital', 'TimbreFiscalDigital')->item(0);
		return $tfd ? $tfd->getAttribute('SelloCFD') : '';
	}

	/**
	 * Extrae el certificado CFD del XML
	 */
	private function extractCertCFD($xml)
	{
		$doc = new DOMDocument();
		$doc->loadXML($xml);
		return $doc->documentElement->getAttribute('Certificado');
	}

	/**
	 * Genera la cadena original
	 */
	private function generateCadenaOriginal($xml)
	{
		global $conf;
		$builder = new CadenaOriginalBuilder($conf->global->CFDIXML_XSLT_PATH);
		return $builder->buildCadenaOriginal($xml);
	}

	/**
	 * Maneja documentos relacionados
	 */
	public function handleRelatedDocuments($object, $stampResult)
	{
		try {
			$invoicetocancel = new Facture($this->db);
			$invoicetocancel->fetch($object->fk_facture_source);

			$uuid = [
				'cancelacion' => $invoicetocancel->array_options['options_cfdixml_UUID'],
				'sustitucion' => $stampResult['uuid']
			];

			$result = $this->cancelDocument($uuid);
			return $this->saveCancellationData($object, $invoicetocancel, $result, $stampResult);
		} catch (Exception $e) {
			dol_syslog("Error en cancelación: " . $e->getMessage(), LOG_ERR);
			return false;
		}
	}

	/**
	 * Guarda los datos de cancelación
	 */
	private function saveCancellationData(&$object, &$invoicetocancel, $result, $stampResult)
	{
		global $conf;
		try {
			// Guardar acuse de cancelación
			$filedir = $conf->facture->multidir_output[$object->entity] . '/' .
				dol_sanitizeFileName($invoicetocancel->ref);

			$xmlFile = $filedir . "/ACUSE_CANCELACION_" . $invoicetocancel->ref . '_' .
				$stampResult['uuid'] . ".xml";

			if (!file_put_contents($xmlFile, utf8_encode($result->Acuse))) {
				throw new Exception('Error guardando acuse de cancelación');
			}

			// Actualizar datos de cancelación en BD
			$sql = "UPDATE " . MAIN_DB_PREFIX . "facture_extrafields SET
                    cfdixml_fechacancelacion = '" . $result->Fecha . "',
                    cfdixml_codigocancelacion = '" . $result->CodigoEstatus . "',
                    cfdixml_xml_cancel = '" . base64_encode($result->Acuse) . "'
                    WHERE fk_object = " . $invoicetocancel->id;

			if ($this->db->query($sql) === false) {
				throw new Exception($this->db->lasterror());
			}

			return true;
		} catch (Exception $e) {
			dol_syslog("Error guardando cancelación: " . $e->getMessage(), LOG_ERR);
			return false;
		}
	}

	/**
	 * Finaliza el proceso de timbrado
	 */
	public function finishStampingProcess($object)
	{
		try {
			// Generar PDF con representación del CFDI
			$object->generateDocument('cfdixmlv2', $this->langs, false, false);

			// Mostrar mensaje de éxito
			setEventMessage('Factura timbrada con éxito UUID:' .
				$object->array_options['options_cfdixml_UUID'], 'mesgs');

			return 1;
		} catch (Exception $e) {
			return $this->handleError('Error finalizando proceso: ' . $e->getMessage());
		}
	}

	/**
	 * Cancela un documento
	 */
	private function cancelDocument($uuid)
	{
		global $conf;

		$client = new SoapClient($conf->global->CFDIXML_WS_CANCEL_URL);
		$response = $client->cancel([
			'uuid' => $uuid,
			'motivo' => '01',
			'username' => strtolower($conf->global->MAIN_INFO_SIREN),
			'password' => $conf->global->CFDIXML_WS_TOKEN
		]);

		if (!$response->cancelResult->Acuse) {
			throw new Exception('Error en cancelación: ' . $response->cancelResult->Mensaje);
		}

		return $response->cancelResult;
	}

	/**
	 * Valida y obtiene los datos de timbrado
	 */
	private function validateAndGetStampingData($parameters)
	{
		$requiredFields = ['uso_cfdi', 'condicion_pago', 'forma_pago', 'metodo_pago', 'exportacion'];

		// Validar campos requeridos
		foreach ($requiredFields as $field) {
			if (!isset($parameters[$field]) || empty($parameters[$field])) {
				return $this->handleError("Campo requerido faltante: $field");
			}
		}

		// Validar reglas de negocio
		if ($parameters['forma_pago'] == '99' && $parameters['metodo_pago'] != 'PPD') {
			return $this->handleError('No se puede tener forma de pago 99 y método de pago PUE');
		}

		if ($parameters['forma_pago'] != '99' && $parameters['metodo_pago'] == 'PPD') {
			return $this->handleError('No se puede tener forma de pago diferente a 99 con PPD');
		}

		return $parameters;
	}

	/**
	 * Actualiza los datos preliminares de la factura
	 */
	private function updateInvoicePreliminaryData(&$object, $data)
	{
		$this->db->begin();

		try {
			// Obtener fecha de factura y hora actual
			$fechaFactura = dol_print_date($object->date, '%Y-%m-%d');
			$horaActual = dol_print_date(dol_now(), '%H:%M:%S');

			$fecha_emision = empty($object->array_options['options_cfdixml_control'])
				? $fechaFactura . 'T' . $horaActual
				: $object->array_options['options_cfdixml_control'];

			// Actualizar datos de CFDI
			$object->array_options['options_cfdixml_usocfdi'] = $data['uso_cfdi'];
			$object->array_options['options_cfdixml_metodopago'] = $data['metodo_pago'];
			$object->array_options['options_cfdixml_exportacion'] = $data['exportacion'];
			$object->array_options['options_cfdixml_control'] = $fecha_emision;

			// Actualizar términos y métodos de pago
			$condicion_pago = dol_getIdFromCode($this->db, $data['metodo_pago'], 'c_payment_term', 'code', 'rowid');
			$forma_pago = dol_getIdFromCode($this->db, $data['forma_pago'], 'c_paiement');

			$object->setPaymentTerms($condicion_pago);
			$object->setPaymentMethods($forma_pago);

			if ($object->update($this->user, 1) < 0) {
				throw new Exception('Error al actualizar la factura');
			}

			$this->db->commit();
			return true;
		} catch (Exception $e) {
			$this->db->rollback();
			return $this->handleError($e->getMessage());
		}
	}

	/**
	 * Maneja errores
	 */
	private function handleError($message)
	{
		setEventMessage($message, 'errors');
		return -1;
	}

	/**
	 * Obtiene forma de pago
	 */
	private function _getFormaPago($id_forma_pago)
	{
		$sql = "SELECT cfp.code FROM " . MAIN_DB_PREFIX . "cfdixml_forma_pago cfp ";
		$sql .= " LEFT JOIN " . MAIN_DB_PREFIX . "c_paiement cp ON cp.id = cfp.fk_c_paiement ";
		$sql .= " WHERE cp.code = '" . $id_forma_pago . '\'';
		$result = $this->db->query($sql);
		$row = $this->db->fetch_object($result);
		return $row->code;
	}

	private function procesarDescripcionesEspecificas($descEspecifica)
	{
		$descripciones = [];

		foreach ($descEspecifica as $desc) {
			$descripciones[] = [
				'Marca' => $desc['Marca'] ?? '',
				'Modelo' => $desc['Modelo'] ?? '',
				'SubModelo' => $desc['SubModelo'] ?? '',
				'NumeroSerie' => $desc['NumeroSerie'] ?? ''
			];
		}

		return $descripciones;
	}

	private function getCountryCode($country)
	{
		$sql = "SELECT code_iso from " . MAIN_DB_PREFIX . "c_country WHERE code = '" . $country . "'";
		$result = $this->db->query($sql);
		$obj = $this->db->fetch_object($result);
		return $obj->code_iso;
	}

	/**
	 * Obtiene los valores de un catálogo desde un archivo SQLite
	 *
	 * @param string $sqlitePath Ruta del archivo SQLite
	 * @param string $tableName Nombre de la tabla en SQLite
	 * @param string $searchColumn Columna donde buscar (opcional)
	 * @param string $searchValue Valor a buscar (opcional)
	 * @param bool $returnSingle Si es true, retorna solo el valor del primer resultado
	 * @param string $valueColumn columna que se usará como value en la respuesta
	 * @param array $labelColumns columnas que se concatenarán para formar el label
	 * @return mixed Array con el catálogo o string con el valor si returnSingle es true
	 */
	private function getCatalogFromSQLite($sqlitePath, $tableName, $searchColumn = '', $searchValue = null, $returnSingle = false, $valueColumn = 'id', $labelColumns = [])
	{
		if (!file_exists($sqlitePath)) {
			throw new RuntimeException("Archivo SQLite no encontrado en la ruta: $sqlitePath");
		}

		try {
			$pdo = new PDO("sqlite:$sqlitePath");
			$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

			// Obtener las columnas de la tabla
			$columnsQuery = "PRAGMA table_info($tableName)";
			$columnsStmt = $pdo->query($columnsQuery);
			$tableColumns = [];
			while ($column = $columnsStmt->fetch(PDO::FETCH_ASSOC)) {
				$tableColumns[] = $column['name'];
			}

			if (empty($tableColumns)) {
				throw new RuntimeException("No se encontraron columnas en la tabla: $tableName");
			}

			// Validar que las columnas existen
			if (!in_array($valueColumn, $tableColumns)) {
				throw new RuntimeException("La columna '$valueColumn' no existe en la tabla");
			}

			// Si no se especifican columnas para el label, usar todas excepto la de valor
			if (empty($labelColumns)) {
				$labelColumns = array_filter($tableColumns, fn($col) => $col !== $valueColumn);
			}

			// Si estamos buscando un valor específico
			if ($searchColumn && $searchValue !== null) {
				if (!in_array($searchColumn, $tableColumns)) {
					throw new RuntimeException("La columna '$searchColumn' no existe en la tabla");
				}

				if ($returnSingle) {
					$query = "SELECT {$valueColumn} FROM $tableName WHERE $searchColumn = :searchValue LIMIT 1";
					$stmt = $pdo->prepare($query);
					$stmt->execute(['searchValue' => $searchValue]);
					$result = $stmt->fetch(PDO::FETCH_COLUMN);
					return $result !== false ? $result : null;
				}

				// Consulta filtrada
				$concatLabels = implode(" || ' - ' || ", $labelColumns);
				$query = "SELECT {$valueColumn} AS value, $concatLabels AS label
                     FROM $tableName
                     WHERE $searchColumn = :searchValue
                     ORDER BY {$valueColumn}";
				$stmt = $pdo->prepare($query);
				$stmt->execute(['searchValue' => $searchValue]);
			} else {
				// Consulta sin filtro
				$concatLabels = implode(" || ' - ' || ", $labelColumns);
				$query = "SELECT {$valueColumn} AS value, $concatLabels AS label
                     FROM $tableName
                     ORDER BY {$valueColumn}";
				$stmt = $pdo->query($query);
			}

			$catalog = [];
			while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
				$catalog[$row['value']] = $row['label'];
			}

			return $catalog;

		} catch (PDOException $e) {
			throw new RuntimeException("Error al consultar el catálogo desde SQLite: " . $e->getMessage());
		}
	}
	/**
	 * Función para depuración del XML
	 */
	public function debugXML($xml)
	{
		try {
			$preparedXML = $this->prepareXMLForStamping($xml);
			echo "<pre>";
			echo "XML Original:\n";
			echo htmlspecialchars($xml) . "\n\n";
			echo "XML Preparado:\n";
			echo htmlspecialchars($preparedXML) . "\n\n";
			echo "XML Base64:\n";
			echo base64_encode($preparedXML) . "\n";
			echo "</pre>";
		} catch (Exception $e) {
			echo "Error: " . $e->getMessage();
		}
	}
}
