<?php
/**
 * \file       cfdixml/class/cfditraslado.class.php
 * \ingroup    cfdixml
 * \brief      Clase para la generación de CFDIs de tipo traslado con complemento Carta Porte
 */

require_once DOL_DOCUMENT_ROOT.'/expedition/class/expedition.class.php';
require_once DOL_DOCUMENT_ROOT.'/product/class/product.class.php';
require_once DOL_DOCUMENT_ROOT.'/societe/class/societe.class.php';
require_once DOL_DOCUMENT_ROOT.'/product/stock/class/entrepot.class.php';

/**
 * Class to manage CFDI for shipping (Traslado type)
 */
class CFDITraslado
{
	/**
	 * @var DoliDB Database handler
	 */
	public $db;

	/**
	 * @var string Error message
	 */
	public $error = '';

	/**
	 * @var array Error messages
	 */
	public $errors = array();

	/**
	 * @var Expedition Shipping object
	 */
	public $shipping;

	/**
	 * @var CFDIManager Instance of CFDI manager
	 */
	public $cfdiManager;

	/**
	 * @var string Generated XML
	 */
	public $xml;

	/**
	 * @var string XML filename
	 */
	public $xmlFilename;

	/**
	 * @var string Path to XML file
	 */
	public $xmlPath;

	/**
	 * Constructor
	 *
	 * @param DoliDB $db Database handler
	 */
	public function __construct($db)
	{
		global $conf;

		$this->db = $db;

		// Cargar la instancia del administrador de CFDI
		dol_include_once('/cfdixml/class/cfdi/CFDIManager.php');
		$this->cfdiManager = new CFDIManager();
	}

	/**
	 * Load shipping data
	 *
	 * @param int $id Shipping ID
	 * @return  int                     <0 if KO, >0 if OK
	 */
	public function fetch($id)
	{
		$this->shipping = new Expedition($this->db);
		$result = $this->shipping->fetch($id);

		if ($result < 0) {
			$this->error = $this->shipping->error;
			$this->errors = $this->shipping->errors;
			return -1;
		}

		// Cargar campos extra y líneas
		if ($result > 0) {
			$this->shipping->fetch_optionals();
			$this->shipping->fetch_lines();
			$this->shipping->fetch_thirdparty();
		}

		return $result;
	}

	/**
	 * Generate CFDI for shipping (Traslado type)
	 *
	 * @param array $params Additional parameters
	 * @return  int                 <0 if KO, >0 if OK
	 */
	public function generate($params = array())
	{
		global $conf, $user, $langs;

		if (empty($this->shipping)) {
			$this->error = 'No se ha cargado el envío';
			return -1;
		}

		try {
			// Crear el directorio de almacenamiento si no existe
			$dir = DOL_DATA_ROOT . '/cfdixml/traslados';
			if (!file_exists($dir)) {
				if (!dol_mkdir($dir)) {
					$this->error = 'Error al crear el directorio para CFDIs de traslado';
					return -1;
				}
			}

			// Preparar los datos para el CFDI
			$datos = $this->prepareDatosCFDI($params);

			// Generar el CFDI
			$debug = !empty($params['debug']) ? true : false;
			$this->xml = $this->cfdiManager->generarCFDI($datos, $debug);

			// Guardar el XML generado
			$result = $this->saveXML();
			if ($result < 0) {
				return -1;
			}

			// Guardar la referencia al XML en el envío (mediante campos personalizados)
			$this->saveReferenceInShipment();

			return 1;
		} catch (Exception $e) {
			$this->error = $e->getMessage();
			dol_syslog('CFDITraslado::generate Error: ' . $this->error, LOG_ERR);
			return -1;
		}
	}

	/**
	 * Prepare data for CFDI
	 *
	 * @param array $params Additional parameters
	 * @return  array               Data for CFDIManager
	 */
	private function prepareDatosCFDI($params)
	{
		global $conf, $mysoc;

		// Cargar datos del destinatario
		$thirdparty = $this->shipping->thirdparty;

		// Obtener datos del emisor (configuración del módulo)
		$emisorRFC = !empty($conf->global->CFDIXML_RFC) ? $conf->global->CFDIXML_RFC : $mysoc->idprof1;
		$emisorNombre = !empty($conf->global->CFDIXML_RAZON_SOCIAL) ? $conf->global->CFDIXML_RAZON_SOCIAL : $mysoc->name;
		$regimenFiscal = !empty($conf->global->CFDIXML_REGIMEN_FISCAL) ? $conf->global->CFDIXML_REGIMEN_FISCAL : '601';
		$lugarExpedicion = !empty($conf->global->CFDIXML_LUGAR_EXPEDICION) ? $conf->global->CFDIXML_LUGAR_EXPEDICION : $mysoc->zip;

		// Rutas de certificados
		$certPath = !empty($conf->global->CFDIXML_CERTIFICADO_PATH) ?
			$conf->global->CFDIXML_CERTIFICADO_PATH :
			DOL_DATA_ROOT . '/cfdixml/certificados/' . $conf->global->CFDIXML_CERTIFICADO;

		$keyPath = !empty($conf->global->CFDIXML_KEY_PATH) ?
			$conf->global->CFDIXML_KEY_PATH :
			DOL_DATA_ROOT . '/cfdixml/certificados/' . $conf->global->CFDIXML_KEY;

		$password = $conf->global->CFDIXML_PASSWORD;

		// Serie y folio para CFDI de traslado
		$serie = !empty($conf->global->CFDIXML_SERIE_TRASLADO) ? $conf->global->CFDIXML_SERIE_TRASLADO : 'T';
		$folio = $this->getNextFolio();

		// Preparar los datos básicos del CFDI
		$datos = array(
			'Certificado' => array(
				'certificadoPath' => $certPath,
				'llavePrivadaPath' => $keyPath,
				'password' => $password
			),
			'Comprobante' => array(
				'Version' => '4.0',
				'Serie' => $serie,
				'Folio' => $folio,
				'Fecha' => date('Y-m-d\TH:i:s'),
				'FormaPago' => null, // No aplica en CFDI de traslado
				'SubTotal' => '0',
				'Moneda' => 'XXX', // Los códigos asignados para las transacciones en que intervenga ninguna moneda
				'Total' => '0',
				'TipoDeComprobante' => 'T',
				'Exportacion' => '01', // No aplica
				'MetodoPago' => null, // No aplica
				'LugarExpedicion' => $lugarExpedicion
			),
			'Emisor' => array(
				'Rfc' => $emisorRFC,
				'Nombre' => $emisorNombre,
				'RegimenFiscal' => $regimenFiscal
			),
			'Receptor' => array(
				'Rfc' => $emisorRFC, // En CFDI de traslado, el RFC del receptor es igual al del emisor
				'Nombre' => $emisorNombre,
				'DomicilioFiscalReceptor' => $lugarExpedicion,
				'RegimenFiscalReceptor' => $regimenFiscal,
				'UsoCFDI' => 'S01' // Sin efectos fiscales
			),
			'Conceptos' => $this->prepareConceptos()
		);

		// Añadir el complemento Carta Porte
		$cartaPorte = $this->prepareCartaPorteData($params);
		if (!empty($cartaPorte)) {
			$datos['Complementos']['CartaPorte'] = $cartaPorte;
		}

		return $datos;
	}

	/**
	 * Prepare conceptos for CFDI
	 *
	 * @return  array       Array of conceptos
	 */
	private function prepareConceptos()
	{
		$conceptos = array();
		$lines = $this->shipping->lines;

		if (empty($lines)) {
			return $conceptos;
		}

		foreach ($lines as $line) {
			// Si existe id de producto, cargar el producto
			if (!empty($line->fk_product)) {
				$product = new Product($this->db);
				$product->fetch($line->fk_product);

				// Obtener código SAT
				$satCode = !empty($product->array_options['options_sat_code']) ?
					$product->array_options['options_sat_code'] :
					'01010101'; // Código genérico

				// Obtener unidad SAT
				$satUnit = !empty($product->array_options['options_sat_unit']) ?
					$product->array_options['options_sat_unit'] :
					'H87'; // Pieza por defecto

				$descripcion = $product->label;
				$unidad = $product->weight_units_label ?: 'Pieza';
			} else {
				// Si no hay producto, usar valores genéricos
				$satCode = '01010101';
				$satUnit = 'H87';
				$descripcion = $line->description ?: 'Producto sin descripción';
				$unidad = 'Pieza';
			}

			// Preparar el concepto
			$concepto = array(
				'ClaveProdServ' => $satCode,
				'Cantidad' => $line->qty,
				'ClaveUnidad' => $satUnit,
				'Unidad' => $unidad,
				'Descripcion' => $descripcion,
				'ValorUnitario' => '0',
				'Importe' => '0',
				'ObjetoImp' => '01' // No objeto de impuesto
			);

			$conceptos[] = $concepto;
		}

		return $conceptos;
	}

	/**
	 * Prepare Carta Porte data
	 *
	 * @param array $params Additional parameters
	 * @return  array               Carta Porte data
	 */
	private function prepareCartaPorteData($params)
	{
		global $conf, $mysoc;

		// Obtener direcciones de origen y destino
		$originAddress = $this->getOriginAddress();
		$destinationAddress = $this->getDestinationAddress();

		// Calcular distancia entre origen y destino (se puede configurar en parámetros)
		$distance = isset($params['distance']) ? floatval($params['distance']) : 100;

		// Calcular peso bruto total
		$totalWeight = $this->calculateTotalWeight();
		if ($totalWeight <= 0) $totalWeight = 1; // Mínimo 1 kg

		// Información del vehículo
		$vehicleInfo = isset($params['vehicle']) ? $params['vehicle'] : array(
			'plate' => !empty($conf->global->CFDIXML_VEHICLE_PLATE) ?
				$conf->global->CFDIXML_VEHICLE_PLATE : 'SIN PLACA',
			'year' => !empty($conf->global->CFDIXML_VEHICLE_YEAR) ?
				$conf->global->CFDIXML_VEHICLE_YEAR : date('Y'),
			'type' => !empty($conf->global->CFDIXML_VEHICLE_TYPE) ?
				$conf->global->CFDIXML_VEHICLE_TYPE : 'C2',
			'insurance' => !empty($conf->global->CFDIXML_INSURANCE) ?
				$conf->global->CFDIXML_INSURANCE : 'ASEGURADORA SA',
			'policy' => !empty($conf->global->CFDIXML_POLICY) ?
				$conf->global->CFDIXML_POLICY : '123456'
		);

		// Información del conductor
		$driverInfo = isset($params['driver']) ? $params['driver'] : array(
			'rfc' => !empty($conf->global->CFDIXML_DRIVER_RFC) ?
				$conf->global->CFDIXML_DRIVER_RFC : $emisorRFC,
			'name' => !empty($conf->global->CFDIXML_DRIVER_NAME) ?
				$conf->global->CFDIXML_DRIVER_NAME : 'CONDUCTOR',
			'license' => !empty($conf->global->CFDIXML_DRIVER_LICENSE) ?
				$conf->global->CFDIXML_DRIVER_LICENSE : 'LICENCIA'
		);

		// Información del permiso SCT
		$permitSCT = !empty($conf->global->CFDIXML_PERMIT_SCT) ?
			$conf->global->CFDIXML_PERMIT_SCT : 'TPAF01';
		$permitNumber = !empty($conf->global->CFDIXML_PERMIT_NUMBER) ?
			$conf->global->CFDIXML_PERMIT_NUMBER : 'PERMISO1234';

		// Datos para el complemento Carta Porte
		$cartaPorte = array(
			'Version' => '3.1',
			'TranspInternac' => isset($params['international']) && $params['international'] ? 'Sí' : 'No',
			'TotalDistRec' => $distance,

			// Ubicaciones
			'Ubicaciones' => array(
				// Origen
				array(
					'TipoUbicacion' => 'Origen',
					'RFCRemitenteDestinatario' => $originAddress['rfc'],
					'NombreRemitenteDestinatario' => $originAddress['name'],
					'FechaHoraSalidaLlegada' => date('Y-m-d\TH:i:s'),
					'Domicilio' => array(
						'Calle' => $originAddress['street'],
						'NumeroExterior' => $originAddress['number'] ?: 'SN',
						'Colonia' => $originAddress['colony'],
						'Localidad' => $originAddress['town'],
						'Municipio' => $originAddress['town'],
						'Estado' => $originAddress['state'],
						'Pais' => 'MEX',
						'CodigoPostal' => $originAddress['zip']
					)
				),
				// Destino
				array(
					'TipoUbicacion' => 'Destino',
					'RFCRemitenteDestinatario' => $destinationAddress['rfc'],
					'NombreRemitenteDestinatario' => $destinationAddress['name'],
					'FechaHoraSalidaLlegada' => date('Y-m-d\TH:i:s', strtotime('+1 day')),
					'DistanciaRecorrida' => $distance,
					'Domicilio' => array(
						'Calle' => $destinationAddress['street'],
						'NumeroExterior' => $destinationAddress['number'] ?: 'SN',
						'Colonia' => $destinationAddress['colony'],
						'Localidad' => $destinationAddress['town'],
						'Municipio' => $destinationAddress['town'],
						'Estado' => $destinationAddress['state'],
						'Pais' => 'MEX',
						'CodigoPostal' => $destinationAddress['zip']
					)
				)
			),

			// Mercancías
			'Mercancias' => array(
				'PesoBrutoTotal' => $totalWeight,
				'UnidadPeso' => 'KGM',
				'NumTotalMercancias' => count($this->shipping->lines),
				'Mercancia' => $this->prepareMercancias()
			),

			// Autotransporte
			'Autotransporte' => array(
				'PermSCT' => $permitSCT,
				'NumPermisoSCT' => $permitNumber,

				// Identificación vehicular
				'IdentificacionVehicular' => array(
					'ConfigVehicular' => $vehicleInfo['type'],
					'PlacaVM' => $vehicleInfo['plate'],
					'AnioModeloVM' => $vehicleInfo['year'],
					'PesoBrutoVehicular' => max(($totalWeight / 1000), 0.5) // Convertir a toneladas, mínimo 0.5
				),

				// Seguros
				'Seguros' => array(
					'AseguraRespCivil' => $vehicleInfo['insurance'],
					'PolizaRespCivil' => $vehicleInfo['policy']
				)
			),

			// Figura de transporte (conductor)
			'FiguraTransporte' => array(
				array(
					'TipoFigura' => '01', // Operador
					'RFCFigura' => $driverInfo['rfc'],
					'NumLicencia' => $driverInfo['license'],
					'NombreFigura' => $driverInfo['name']
				)
			)
		);

		return $cartaPorte;
	}

	/**
	 * Prepare mercancías data
	 *
	 * @return  array   Array of mercancías
	 */
	private function prepareMercancias()
	{
		$mercancias = array();
		$lines = $this->shipping->lines;

		if (empty($lines)) {
			return $mercancias;
		}

		foreach ($lines as $line) {
			// Si existe id de producto, cargar el producto
			if (!empty($line->fk_product)) {
				$product = new Product($this->db);
				$product->fetch($line->fk_product);

				// Obtener código SAT
				$satCode = !empty($product->array_options['options_sat_code']) ?
					$product->array_options['options_sat_code'] :
					'01010101'; // Código genérico

				// Obtener unidad SAT
				$satUnit = !empty($product->array_options['options_sat_unit']) ?
					$product->array_options['options_sat_unit'] :
					'H87'; // Pieza por defecto

				$descripcion = $product->label;
				$unidad = $product->weight_units_label ?: 'Pieza';

				// Calcular peso
				$weight = !empty($product->weight) ? $product->weight * $line->qty : 1;

				// Verificar si es material peligroso
				$materialPeligroso = !empty($product->array_options['options_material_peligroso']) ? 'Sí' : 'No';
			} else {
				// Si no hay producto, usar valores genéricos
				$satCode = '01010101';
				$satUnit = 'H87';
				$descripcion = $line->description ?: 'Producto sin descripción';
				$unidad = 'Pieza';
				$weight = 1; // Peso mínimo
				$materialPeligroso = 'No';
			}

			// Preparar la mercancía
			$mercancia = array(
				'BienesTransp' => $satCode,
				'Descripcion' => $descripcion,
				'Cantidad' => $line->qty,
				'ClaveUnidad' => $satUnit,
				'Unidad' => $unidad,
				'MaterialPeligroso' => $materialPeligroso,
				'PesoEnKg' => $weight
			);

			// Si es material peligroso, agregar datos adicionales
			if ($materialPeligroso == 'Sí' && !empty($product->array_options['options_clave_material_peligroso'])) {
				$mercancia['CveMaterialPeligroso'] = $product->array_options['options_clave_material_peligroso'];
				$mercancia['Embalaje'] = $product->array_options['options_embalaje'] ?: '4D';
				$mercancia['DescripEmbalaje'] = $product->array_options['options_descripcion_embalaje'] ?: 'Cajas de madera contrachapada';
			}

			$mercancias[] = $mercancia;
		}

		return $mercancias;
	}

	/**
	 * Get origin address
	 *
	 * @return  array   Origin address data
	 */
	private function getOriginAddress()
	{
		global $conf, $mysoc;

		// Obtener RFC del emisor
		$emisorRFC = !empty($conf->global->CFDIXML_RFC) ? $conf->global->CFDIXML_RFC : $mysoc->idprof1;

		// Si hay un almacén de origen
		if (!empty($this->shipping->entrepot_id)) {
			$warehouse = new Entrepot($this->db);
			$warehouse->fetch($this->shipping->entrepot_id);

			return array(
				'rfc' => $emisorRFC,
				'name' => $warehouse->label,
				'street' => $warehouse->address,
				'number' => $this->extractNumber($warehouse->address),
				'colony' => $warehouse->town, // Simplificado
				'town' => $warehouse->town,
				'state' => $this->getStateCode($warehouse->state_id),
				'zip' => $warehouse->zip
			);
		}

		// Por defecto, usar los datos de la empresa
		return array(
			'rfc' => $emisorRFC,
			'name' => $mysoc->name,
			'street' => $mysoc->address,
			'number' => $this->extractNumber($mysoc->address),
			'colony' => $mysoc->town, // Simplificado
			'town' => $mysoc->town,
			'state' => $this->getStateCode($mysoc->state_id),
			'zip' => $mysoc->zip
		);
	}

	/**
	 * Get destination address
	 *
	 * @return  array   Destination address data
	 */
	private function getDestinationAddress()
	{
		// Obtener datos del tercero
		$thirdparty = $this->shipping->thirdparty;

		// Por defecto, usar dirección del tercero
		$street = $thirdparty->address;
		$town = $thirdparty->town;
		$state_id = $thirdparty->state_id;
		$zip = $thirdparty->zip;

		// Si hay dirección de entrega
		if (!empty($this->shipping->fk_delivery_address)) {
			require_once DOL_DOCUMENT_ROOT . '/core/class/address.class.php';
			$deliveryAddress = new Contact($this->db);
			$deliveryAddress->fetch($this->shipping->fk_delivery_address);

			$street = $deliveryAddress->address;
			$town = $deliveryAddress->town;
			$state_id = $deliveryAddress->state_id;
			$zip = $deliveryAddress->zip;
		}

		return array(
			'rfc' => $thirdparty->idprof1,
			'name' => $thirdparty->name,
			'street' => $street,
			'number' => $this->extractNumber($street),
			'colony' => $town, // Simplificado
			'town' => $town,
			'state' => $this->getStateCode($state_id),
			'zip' => $zip
		);
	}

	/**
	 * Get state code from state ID
	 *
	 * @param int $state_id State ID
	 * @return  string               State code for SAT
	 */
	private function getStateCode($state_id)
	{
		global $conf;

		// Mapping de estados de Dolibarr a códigos SAT
		$stateMappings = array(
			// Aquí deberías colocar el mapeo completo entre IDs de estados de Dolibarr y códigos de estado del SAT
			// Ejemplo:
			// 1 => 'AGU', // Aguascalientes
			// 2 => 'BCN', // Baja California Norte
			// etc.
		);

		// Si existe el mapeo, retornarlo
		if (isset($stateMappings[$state_id])) {
			return $stateMappings[$state_id];
		}

		// Si no existe, buscar en configuración o usar un valor por defecto
		return !empty($conf->global->CFDIXML_DEFAULT_STATE) ? $conf->global->CFDIXML_DEFAULT_STATE : 'CMX';
	}

	/**
	 * Extract number from street address
	 *
	 * @param string $address Address
	 * @return  string              Number extracted from address
	 */
	private function extractNumber($address)
	{
		// Intentar extraer el número de la dirección (implementación básica)
		if (preg_match('/(\d+)/', $address, $matches)) {
			return $matches[1];
		}

		return 'SN'; // Sin número
	}

	/**
	 * Calculate total weight
	 *
	 * @return  float   Total weight in kg
	 */
	private function calculateTotalWeight()
	{
		$totalWeight = 0;
		$lines = $this->shipping->lines;

		if (empty($lines)) {
			return $totalWeight;
		}

		foreach ($lines as $line) {
			if (!empty($line->fk_product)) {
				$product = new Product($this->db);
				$product->fetch($line->fk_product);

				if (!empty($product->weight)) {
					$totalWeight += $product->weight * $line->qty;
				} else {
					// Si no tiene peso, asignar 1kg por unidad
					$totalWeight += 1 * $line->qty;
				}
			} else {
				// Si no hay producto, asignar 1kg por unidad
				$totalWeight += 1 * $line->qty;
			}
		}

		return $totalWeight > 0 ? $totalWeight : 1; // Mínimo 1kg
	}

	/**
	 * Get next folio number
	 *
	 * @return  string  Next folio number
	 */
	private function getNextFolio()
	{
		global $conf;

		$folio = 1; // Valor predeterminado

		// Obtener el último folio usado
		if (!empty($conf->global->CFDIXML_LAST_FOLIO_TRASLADO)) {
			$folio = $conf->global->CFDIXML_LAST_FOLIO_TRASLADO + 1;
		}

		// Actualizar el contador
		dolibarr_set_const($this->db, 'CFDIXML_LAST_FOLIO_TRASLADO', $folio, 'chaine', 0, '', $conf->entity);

		// Formatear el folio con ceros a la izquierda
		return sprintf('%08d', $folio);
	}

	/**
	 * Save XML file
	 *
	 * @return  int     <0 if KO, >0 if OK
	 */
	private function saveXML()
	{
		global $conf;

		// Crear el nombre del archivo
		$this->xmlFilename = 'CFDI_Traslado_' . $this->shipping->ref . '_' . date('YmdHis') . '.xml';
		$this->xmlPath = DOL_DATA_ROOT . '/cfdixml/traslados/' . $this->xmlFilename;

		// Guardar el archivo
		if (!file_put_contents($this->xmlPath, $this->xml)) {
			$this->error = 'Error al guardar el archivo XML';
			return -1;
		}

		return 1;
	}

	/**
	 * Save reference to CFDI in shipment
	 *
	 * @return  int     <0 if KO, >0 if OK
	 */
	private function saveReferenceInShipment()
	{
		global $conf;

		// Verificar si ya existe el campo extra en expedition
		include_once DOL_DOCUMENT_ROOT . '/core/class/extrafields.class.php';
		$extrafields = new ExtraFields($this->db);
		$extrafieldslabels = $extrafields->fetch_name_optionals_label('expedition');

		// Crear el campo si no existe
		if (!isset($extrafieldslabels['cfdi_traslado'])) {
			$result = $extrafields->addExtraField(
				'cfdi_traslado',            // Atributo
				'CFDI de Traslado',         // Etiqueta
				'varchar',                  // Tipo
				'varchar',                  // Tipo
				100,                        // Posición
				255,                        // Tamaño
				'expedition',               // Elemento
				0,                          // Único
				0,                          // Requerido
				'',                         // Valor por defecto
				array('position' => 100)    // Opciones
			);

			if ($result < 0) {
				$this->error = 'Error al crear el campo personalizado para CFDI de traslado';
				return -1;
			}
		}

		// Guardar la referencia al XML
		$this->shipping->array_options['options_cfdi_traslado'] = $this->xmlFilename;
		$result = $this->shipping->insertExtraFields();

		if ($result < 0) {
			$this->error = $this->shipping->error;
			return -1;
		}

		return 1;
	}

	/**
	 * Get download URL for XML
	 *
	 * @return string URL to download the XML file
	 */
	public function getXmlDownloadUrl()
	{
		global $conf;

		// Verificar si hay un XML generado
		if (empty($this->xmlFilename)) {
			// Si no se generó en esta instancia, verificar si existe en el envío
			$this->xmlFilename = $this->shipping->array_options['options_cfdi_traslado'] ?? '';

			if (empty($this->xmlFilename)) {
				return '';
			}
		}

		// Construir la URL de descarga
		return DOL_URL_ROOT . '/cfdixml/document.php?type=traslado&file=' . urlencode($this->xmlFilename);
	}

	/**
	 * Check if shipment has CFDI Traslado
	 *
	 * @return boolean True if has CFDI, false otherwise
	 */
	public function hasCfdiTraslado()
	{
		// Verificar si el envío tiene un CFDI de traslado
		if (!empty($this->shipping->array_options['options_cfdi_traslado'])) {
			$xmlFile = DOL_DATA_ROOT . '/cfdixml/traslados/' . $this->shipping->array_options['options_cfdi_traslado'];
			return file_exists($xmlFile);
		}

		return false;
	}

	/**
	 * Get XML content
	 *
	 * @return string XML content or empty string if not found
	 */
	public function getXmlContent()
	{
		// Si ya tenemos el XML en memoria, devolverlo
		if (!empty($this->xml)) {
			return $this->xml;
		}

		// Si no, intentar leerlo desde el archivo
		if (!empty($this->shipping->array_options['options_cfdi_traslado'])) {
			$xmlFile = DOL_DATA_ROOT . '/cfdixml/traslados/' . $this->shipping->array_options['options_cfdi_traslado'];
			if (file_exists($xmlFile)) {
				return file_get_contents($xmlFile);
			}
		}

		return '';
	}

	/**
	 * Cancel CFDI Traslado
	 *
	 * @param string $motivo Motivo de cancelación
	 * @return int <0 if KO, >0 if OK
	 */
	public function cancel($motivo = '02')
	{
		global $conf, $user;

		// Verificar si existe un CFDI de traslado
		if (!$this->hasCfdiTraslado()) {
			$this->error = 'No existe un CFDI de traslado para este envío';
			return -1;
		}

		// Implementar lógica de cancelación de CFDI con el PAC
		// Esta parte dependerá de cómo implementes la cancelación de CFDI en tu módulo

		// Como ejemplo simplificado:
		$cancelado = true; // Aquí iría la lógica real de cancelación

		if ($cancelado) {
			// Marcar como cancelado en el envío
			$this->shipping->array_options['options_cfdi_traslado_cancelado'] = 1;
			$result = $this->shipping->insertExtraFields();

			if ($result < 0) {
				$this->error = $this->shipping->error;
				return -1;
			}

			// Registrar la acción en el historial
			require_once DOL_DOCUMENT_ROOT . '/core/class/actioncomm.class.php';
			$actioncomm = new ActionComm($this->db);
			$actioncomm->type_code = 'AC_OTH_AUTO';
			$actioncomm->label = 'CFDI de traslado cancelado: ' . $this->shipping->array_options['options_cfdi_traslado'];
			$actioncomm->note_private = 'Motivo de cancelación: ' . $motivo;
			$actioncomm->fk_element = $this->shipping->id;
			$actioncomm->elementtype = 'shipping';
			$actioncomm->userownerid = $user->id;

			$result = $actioncomm->add($user);

			return 1;
		} else {
			$this->error = 'Error al cancelar el CFDI de traslado';
			return -1;
		}
	}
}
