<?php
/**
 * \file       cfdixml/class/cfdi/ComercioExteriorManager.php
 * \ingroup    cfdixml
 * \brief      Clase para el manejo del complemento de Comercio Exterior
 */

class ComercioExteriorManager extends AbstractComplementoManager
{
	/** @var string Versión del complemento */
	private const VERSION = '2.0';

	/**
	 * Genera el complemento de Comercio Exterior
	 *
	 * @param array $datos Datos del complemento
	 * @return object Complemento generado
	 */
	public function generarComplemento(array $datos)
	{
		if (!$this->validarDatos($datos)) {
			throw new Exception("Los datos proporcionados no son válidos para el complemento de Comercio Exterior");
		}

		try {
			// Crear instancia del complemento usando cfdiutils
			$this->complemento = new \CfdiUtils\Elements\ComercioExterior20\ComercioExterior([
				'Version' => self::VERSION,
				'MotivoTraslado' => $datos['MotivoTraslado'] ?? null,
				'TipoOperacion' => $datos['TipoOperacion'] ?? '2',  // 2 = Exportación
				'ClaveDePedimento' => $datos['ClaveDePedimento'] ?? 'A1',
				'CertificadoOrigen' => $datos['CertificadoOrigen'] ?? '0',
				'NumCertificadoOrigen' => $datos['NumCertificadoOrigen'] ?? null,
				'NumeroExportadorConfiable' => $datos['NumeroExportadorConfiable'] ?? null,
				'Incoterm' => $datos['Incoterm'] ?? null,
				'Subdivision' => $datos['Subdivision'] ?? null,
				'TipoCambioUSD' => $datos['TipoCambioUSD'],
				'TotalUSD' => $datos['TotalUSD']
			]);

			// Procesar emisor si existe
			if (isset($datos['Emisor'])) {
				$this->agregarEmisor($datos['Emisor']);
			}

			// Procesar receptor si existe
			if (isset($datos['Receptor'])) {
				$this->agregarReceptor($datos['Receptor']);
			}

			// Procesar destinatario si existe
			if (isset($datos['Destinatario'])) {
				$this->agregarDestinatario($datos['Destinatario']);
			}

			// Procesar mercancías (obligatorio)
			$this->agregarMercancias($datos['Mercancias']);

			return $this->complemento;

		} catch (Exception $e) {
			throw new Exception("Error al generar complemento de Comercio Exterior: " . $e->getMessage());
		}
	}

	/**
	 * Valida los datos específicos del complemento
	 */
	protected function validarDatos(array $datos): bool
	{
		// Validar campos obligatorios base
		$obligatorios = ['TipoCambioUSD', 'TotalUSD', 'Mercancias'];
		foreach ($obligatorios as $campo) {
			if (!isset($datos[$campo])) {
				throw new Exception("Falta el campo obligatorio: $campo");
			}
		}

		// Validar que existan mercancías
		if (empty($datos['Mercancias'])) {
			throw new Exception("Debe especificar al menos una mercancía");
		}

		// Validar coherencia entre tipo de operación y complemento
		if (isset($datos['TipoOperacion']) && $datos['TipoOperacion'] !== '2') {
			throw new Exception("El tipo de operación debe ser '2' (Exportación)");
		}

		// Validar congruencia de datos monetarios
		if (isset($datos['TotalUSD']) && $datos['TotalUSD'] <= 0) {
			throw new Exception("El total en USD debe ser mayor a 0");
		}

		return true;
	}

	/**
	 * Agrega la información del emisor al complemento
	 */
	private function agregarEmisor(array $datos)
	{
		$emisor = [];

		// Agregar CURP solo si es persona física
		if (isset($datos['Curp'])) {
			$emisor['Curp'] = $datos['Curp'];
		}

		// Procesar domicilio del emisor
		if (isset($datos['Domicilio'])) {
			$emisor['Domicilio'] = $this->procesarDomicilio($datos['Domicilio']);
		}

		$this->complemento->addEmisor($emisor);
	}

	/**
	 * Agrega la información del receptor al complemento
	 */
	private function agregarReceptor(array $datos)
	{
		$receptor = [];

		if (isset($datos['NumRegIdTrib'])) {
			$receptor['NumRegIdTrib'] = $datos['NumRegIdTrib'];
		}

		if (isset($datos['Domicilio'])) {
			$receptor['Domicilio'] = $this->procesarDomicilio($datos['Domicilio']);
		}

		$this->complemento->addReceptor($receptor);
	}

	/**
	 * Agrega la información del destinatario al complemento
	 */
	private function agregarDestinatario(array $datos)
	{
		$destinatario = [];

		if (isset($datos['NumRegIdTrib'])) {
			$destinatario['NumRegIdTrib'] = $datos['NumRegIdTrib'];
		}
		if (isset($datos['Nombre'])) {
			$destinatario['Nombre'] = $datos['Nombre'];
		}
		if (isset($datos['Domicilio'])) {
			$destinatario['Domicilio'] = $this->procesarDomicilio($datos['Domicilio']);
		}

		$this->complemento->addDestinatario($destinatario);
	}

	/**
	 * Procesa y agrega las mercancías al complemento
	 */
	private function agregarMercancias(array $mercancias)
	{
		$mercanciasProcesadas = [];

		foreach ($mercancias as $mercancia) {
			$mercanciaData = [
				'NoIdentificacion' => $mercancia['NoIdentificacion'],
				'FraccionArancelaria' => $mercancia['FraccionArancelaria'],
				'CantidadAduana' => $mercancia['CantidadAduana'],
				'UnidadAduana' => $mercancia['UnidadAduana'],
				'ValorUnitarioAduana' => $mercancia['ValorUnitarioAduana'],
				'ValorDolares' => $mercancia['ValorDolares']
			];

			if (isset($mercancia['DescripcionesEspecificas'])) {
				foreach ($mercancia['DescripcionesEspecificas'] as $desc) {
					$mercanciaData['DescripcionesEspecificas'][] = $this->procesarDescripcionEspecifica($desc);
				}
			}

			$mercanciasProcesadas[] = $mercanciaData;
		}

		$this->complemento->addMercancias(['Mercancia' => $mercanciasProcesadas]);
	}

	/**
	 * Procesa la información de domicilio
	 */
	private function procesarDomicilio(array $datos): array
	{
		$domicilio = [];
		$camposDomicilio = [
			'Calle', 'NumeroExterior', 'NumeroInterior', 'Colonia',
			'Localidad', 'Referencia', 'Municipio', 'Estado', 'Pais', 'CodigoPostal'
		];

		foreach ($camposDomicilio as $campo) {
			if (isset($datos[$campo])) {
				$domicilio[$campo] = $datos[$campo];
			}
		}

		return $domicilio;
	}

	/**
	 * Procesa la descripción específica de una mercancía
	 */
	private function procesarDescripcionEspecifica(array $desc): array
	{
		return [
			'Marca' => $desc['Marca'],
			'Modelo' => $desc['Modelo'] ?? '',
			'SubModelo' => $desc['SubModelo'] ?? '',
			'NumeroSerie' => $desc['NumeroSerie'] ?? ''
		];
	}
}
