<?php
/**
 * \file       cfdixml/class/cfdi/AbstractComplementoManager.php
 * \ingroup    cfdixml
 * \brief      Clase base abstracta para el manejo de complementos CFDI
 */

abstract class AbstractComplementoManager
{
	/** @var array Datos del CFDI principal */
	protected $cfdi;

	/** @var object Instancia del complemento específico usando cfdiutils */
	protected $complemento;

	/**
	 * Constructor
	 *
	 * @param array $cfdi Datos del CFDI principal
	 */
	public function __construct($cfdi)
	{
		$this->cfdi = $cfdi;
	}

	/**
	 * Método abstracto para generar el complemento
	 * Cada implementación específica debe implementar su propia lógica
	 *
	 * @param array $datos Datos específicos del complemento
	 * @return mixed Complemento generado
	 */
	abstract public function generarComplemento(array $datos);

	/**
	 * Método abstracto para validar los datos del complemento
	 *
	 * @param array $datos Datos a validar
	 * @return bool True si los datos son válidos
	 */
	abstract protected function validarDatos(array $datos): bool;

	/**
	 * Obtiene el complemento generado
	 *
	 * @return object|null
	 */
	public function getComplemento()
	{
		return $this->complemento;
	}

	/**
	 * Método para aplicar el complemento al CFDI
	 *
	 * @param \CfdiUtils\CfdiCreator40 $creator Instancia del creador de CFDI
	 * @return void
	 * @throws Exception si no hay complemento generado
	 */
	public function aplicarComplemento($creator)
	{
		if (!$this->complemento) {
			throw new Exception("No hay complemento generado para aplicar al CFDI");
		}

		$creator->comprobante()->addComplemento($this->complemento);
	}
}
