
<?php
/**
 * \file       cfdixml/src/CFDIManager.php
 * \ingroup    cfdixml
 * \brief      Clase para la generación de CFDIs usando cfdiutils
 */
dol_include_once('/cfdixml/vendor/autoload.php');

use CfdiUtils\CfdiCreator40;
use CfdiUtils\Certificado\Certificado;
use PhpCfdi\Credentials\Credential;

class CFDIManager
{
	private $xmlResolver;
	private $basePath;

	public function __construct()
	{
		try {
			// Configurar la ruta base
			$this->basePath = dol_buildpath('/cfdixml/resources-sat-xml/resources/www.sat.gob.mx/sitio_internet', 0);

			// Verificar que existe el directorio base
			if (!is_dir($this->basePath)) {
				throw new Exception("No se encuentra el directorio base de recursos SAT: " . $this->basePath);
			}

			// Verificar que existen los archivos necesarios
			$this->verificarRecursos();

			// Inicializar y configurar XmlResolver
			$this->xmlResolver = new \CfdiUtils\XmlResolver\XmlResolver();
			$this->xmlResolver->setLocalPath($this->basePath . '/cfd/4');
		} catch (Exception $e) {
			throw new Exception("Error inicializando CFDIManager: " . $e->getMessage());
		}
	}

	private function verificarRecursos()
	{
		// Verificar XSD
		$xsdPath = $this->basePath . '/cfd/4/cfdv40.xsd';
		if (!file_exists($xsdPath)) {
			throw new Exception("No se encuentra el archivo XSD en: " . $xsdPath);
		}

		// Verificar XSLT
		$xsltPath = $this->basePath . '/cfd/4/cadenaoriginal_4_0/cadenaoriginal_4_0.xslt';
		if (!file_exists($xsltPath)) {
			throw new Exception("No se encuentra el archivo XSLT en: " . $xsltPath);
		}
	}
	//	public function generarCFDI($datos, $debug = false)
	//	{
	//		try {
	//			// Validar estructura de datos requerida
	//			$this->validarDatosRequeridos($datos);
	//
	//			// Crear las credenciales
	//			$credential = Credential::openFiles(
	//				$datos['Certificado']['certificadoPath'],
	//				$datos['Certificado']['llavePrivadaPath'],
	//				$datos['Certificado']['password']
	//			);
	//
	//			// Inicializar el creador de CFDI
	//			$creator = new CfdiCreator40($datos['Comprobante']);
	//
	//			if ($this->xmlResolver) {
	//				$creator->setXmlResolver($this->xmlResolver);
	//			}
	//
	//			// Establecer el certificado
	//			$creator->putCertificado(
	//				new Certificado($credential->certificate()->pem()),
	//				false // No establecer automáticamente RFC y nombre del emisor
	//			);
	//
	//			$comprobante = $creator->comprobante();
	//
	//			// Agregar Emisor
	//			$comprobante->addEmisor($datos['Emisor']);
	//
	//			// Agregar Receptor
	//			$comprobante->addReceptor($datos['Receptor']);
	//
	//			// Agregar Conceptos con sus impuestos
	//			foreach ($datos['Conceptos'] as $conceptoData) {
	//				$this->agregarConceptoConImpuestos($comprobante, $conceptoData);
	//			}
	//
	//			// Calcular totales
	//			$creator->addSumasConceptos(null, 2);
	//
	//			// Agregar sello
	//			$creator->addSello(
	//				$credential->privateKey()->pem(),
	//				$credential->privateKey()->passPhrase()
	//			);
	//
	//			// Mover definiciones al comprobante
	//			$creator->moveSatDefinitionsToComprobante();
	//
	//			// Obtener el XML
	//			$xml = $creator->asXml();
	//
	//			// Mostrar información de debug si está activado
	//			if ($debug) {
	//				echo "<pre>";
	//				echo "=== XML Generado ===\n";
	//				echo htmlspecialchars($xml);
	//				echo "\n\n";
	//
	//				// Validar y mostrar errores si existen
	//				$asserts = $creator->validate();
	//				if ($asserts->hasErrors()) {
	//					echo "=== Errores de Validación ===\n";
	//					print_r($asserts->errors());
	//				}
	//				echo "</pre>";
	//			}
	//
	//			// Validar el CFDI
	//			$asserts = $creator->validate();
	//
	//			if ($asserts->hasErrors()) {
	//				throw new Exception("Errores en la validación del CFDI: " . implode(", ", $asserts->errors()));
	//			}
	//
	//			return $xml;
	//
	//		} catch (Exception $e) {
	//			throw new Exception("Error generando CFDI: " . $e->getMessage());
	//		}
	//	}

	public function generarCFDI($datos, $debug = false)
	{
		try {
			// Validar estructura de datos requerida
			$this->validarDatosRequeridos($datos);

			// Crear las credenciales
			$credential = Credential::openFiles(
				$datos['Certificado']['certificadoPath'],
				$datos['Certificado']['llavePrivadaPath'],
				$datos['Certificado']['password']
			);

			// Inicializar el creador de CFDI
			$creator = new CfdiCreator40($datos['Comprobante']);

			if ($this->xmlResolver) {
				// Si hay complemento de comercio exterior, agregar su ruta
				if (
					isset($datos['Complementos']['ComercioExterior']) &&
					$datos['Comprobante']['Exportacion'] === '02'
				) {


					$this->xmlResolver->setLocalPath($this->basePath . '/cfd/ComercioExterior20');
					if ($debug) {
						dol_syslog("Configurando ruta CCE: " . $this->basePath . '/cfd/ComercioExterior20', LOG_DEBUG);
					}
				}
				$creator->setXmlResolver($this->xmlResolver);
			}

			// Establecer el certificado
			$creator->putCertificado(
				new Certificado($credential->certificate()->pem()),
				false // No establecer automáticamente RFC y nombre del emisor
			);

			$comprobante = $creator->comprobante();

			$datos['Emisor']['Rfc'] = $this->normalizarCadenaRFC($datos['Emisor']['Rfc']);
			// Agregar Emisor
			$comprobante->addEmisor($datos['Emisor']);

			// Agregar Receptor
			$comprobante->addReceptor($datos['Receptor']);

			// Agregar Conceptos con sus impuestos
			foreach ($datos['Conceptos'] as $conceptoData) {
				//				echo '<pre>';print_r($conceptoData);exit;
				$this->agregarConceptoConImpuestos($comprobante, $conceptoData);
			}

			// **Agregar el complemento Comercio Exterior**
			if (isset($datos['Complementos']['ComercioExterior'])) {
				try {
					$this->agregarComplementoComercioExterior($creator, $datos['Complementos']['ComercioExterior']);
					if ($debug) {
						echo "Complemento de Comercio Exterior agregado correctamente\n";
					}
				} catch (Exception $e) {
					throw new Exception("Error al agregar complemento de comercio exterior: " . $e->getMessage());
				}
			}

			// Calcular totales
			$creator->addSumasConceptos(null, 2);
			//			echo '<pre>';print_r($datos);exit;
			// Agregar sello
			$creator->addSello(
				$credential->privateKey()->pem(),
				$credential->privateKey()->passPhrase()
			);

			// Mover definiciones al comprobante
			$creator->moveSatDefinitionsToComprobante();

			// Obtener el XML
			$xml = $creator->asXml();

			// Mostrar información de debug si está activado
			if ($debug) {
				echo "<pre>";
				echo "=== XML Generado ===\n";
				echo htmlspecialchars($xml);
				echo "\n\n";

				// Validar y mostrar errores si existen
				$asserts = $creator->validate();
				if ($asserts->hasErrors()) {
					echo "=== Errores de Validación ===\n";
					print_r($asserts->errors());
				}
				echo "</pre>";
			}

			// Validar el CFDI
			$asserts = $creator->validate();

			if ($asserts->hasErrors()) {
				throw new Exception("Errores en la validación del CFDI: " . implode(", ", $asserts->errors()));
			}

			return $xml;
		} catch (Exception $e) {
			throw new Exception("Error generando CFDI: " . $e->getMessage());
		}
	}

	private function agregarComplementoComercioExterior(CfdiCreator40 $creator, array $comercioExteriorData)
	{
		try {
			// Crear el complemento
			$cce20 = new \CfdiUtils\Elements\Cce20\ComercioExterior([
				'Version' => $comercioExteriorData['Version'],
				'MotivoTraslado' => $comercioExteriorData['MotivoTraslado'],
				'ClaveDePedimento' => $comercioExteriorData['ClaveDePedimento'],
				'CertificadoOrigen' => $comercioExteriorData['CertificadoOrigen'],
				'NumCertificadoOrigen' => $comercioExteriorData['NumCertificadoOrigen'],
				'NumExportadorConfiable' => $comercioExteriorData['NumExportadorConfiable'],
				'Incoterm' => $comercioExteriorData['Incoterm'],
				'Observaciones' => $comercioExteriorData['Observaciones'],
				'TipoCambioUSD' => $comercioExteriorData['TipoCambioUSD'],
				'TotalUSD' => $comercioExteriorData['TotalUSD']
			]);

			// Agregar Emisor
			$emisor = $cce20->getEmisor();
			$emisor->addCurp($comercioExteriorData['Emisor']['Curp']); //Quitar solo para pruebas
			$emisor->addDomicilio($comercioExteriorData['Emisor']['Domicilio']);

			// Agregar Receptor
			$receptor = $cce20->addReceptor([
				'NumRegIdTrib' => $comercioExteriorData['Receptor']['NumRegIdTrib']
			]);
			$receptor->addDomicilio($comercioExteriorData['Receptor']['Domicilio']);

			// Agregar Mercancías
			$mercancias = $cce20->getMercancias();
			foreach ($comercioExteriorData['Mercancias'] as $mercanciaData) {
				$mercancias->addMercancia([
					'NoIdentificacion' => $mercanciaData['NoIdentificacion'],
					'FraccionArancelaria' => $mercanciaData['FraccionArancelaria'],
					'CantidadAduana' => $mercanciaData['CantidadAduana'],
					'UnidadAduana' => $mercanciaData['UnidadAduana'],
					'ValorUnitarioAduana' => $mercanciaData['ValorUnitarioAduana'],
					'ValorDolares' => $mercanciaData['ValorDolares']
				]);
			}

			// Agregar el complemento al comprobante
			$creator->comprobante()->addComplemento($cce20);

			return true;
		} catch (Exception $e) {
			dol_syslog("Error en agregarComplementoComercioExterior: " . $e->getMessage(), LOG_ERR);
			throw new Exception("Error agregando complemento de comercio exterior: " . $e->getMessage());
		}
	}

	private function agregarConceptoConImpuestos($comprobante, $conceptoData)
	{
		// Separar impuestos de los datos del concepto
		$impuestos = isset($conceptoData['Impuestos']) ? $conceptoData['Impuestos'] : null;
		$datosConcepto = $this->filtrarAtributosConcepto($conceptoData);
		//		echo '<pre>';print_r($impuestos);exit;
		// Agregar concepto base
		$concepto = $comprobante->addConcepto($datosConcepto);

		// Agregar impuestos solo si existen
		if ($impuestos) {
			// Agregar traslados si existen y no están vacíos
			if (!empty($impuestos['Traslados']) && is_array($impuestos['Traslados'])) {
				foreach ($impuestos['Traslados'] as $traslado) {
					if (!empty($traslado)) {  // Verificar que el traslado tenga datos
						$concepto->addTraslado($traslado);
					}
				}
			}

			// Agregar retenciones solo si existen y no están vacías
			if (!empty($impuestos['Retenciones']) && is_array($impuestos['Retenciones'])) {
				foreach ($impuestos['Retenciones'] as $retencion) {
					if (!empty($retencion)) {  // Verificar que la retención tenga datos
						$concepto->addRetencion($retencion);
					}
				}
			}
		}
	}

	private function filtrarAtributosConcepto($concepto)
	{
		$conceptoFiltrado = $concepto;
		unset($conceptoFiltrado['Impuestos']);
		return $conceptoFiltrado;
	}

	private function validarDatosRequeridos($datos)
	{
		$requeridos = ['Comprobante', 'Emisor', 'Receptor', 'Conceptos', 'Certificado'];

		foreach ($requeridos as $campo) {
			if (!isset($datos[$campo])) {
				throw new Exception("Falta el campo requerido: $campo");
			}
		}

		// Validar datos del certificado
		$certRequeridos = ['certificadoPath', 'llavePrivadaPath', 'password'];
		foreach ($certRequeridos as $campo) {
			if (!isset($datos['Certificado'][$campo])) {
				throw new Exception("Falta el campo requerido en Certificado: $campo");
			}
		}
	}

	/**
	 * Normaliza una cadena reemplazando caracteres compuestos (como Ñ) por sus equivalentes precompuestos (como Ñ)
	 *
	 * @param string $texto
	 * @return string
	 */
	function normalizarCadenaRFC($texto)
	{
		// Mapa de caracteres compuestos -> precompuestos
		$reemplazos = [
			'N' . "\u{0303}" => 'Ñ', // N + ̃
			'n' . "\u{0303}" => 'ñ', // n + ̃
		];

		// Reemplazo manual usando strtr
		return strtr($texto, $reemplazos);
	}
}
