<?php

/**
 * Clase para gestionar las addendas del módulo CFDIXML
 */
class CfdixmlAddenda
{
	/**
	 * @var DoliDB Database handler
	 */
	public $db;

	/**
	 * @var string Error string
	 */
	public $error;

	/**
	 * @var array Errors array
	 */
	public $errors = array();

	/**
	 * Constructor
	 *
	 * @param DoliDB $db Database handler
	 */
	public function __construct($db)
	{
		$this->db = $db;
	}

	/**
	 * Actualiza el estado de una addenda
	 *
	 * @param int $id ID de la addenda
	 * @param int $status Nuevo estado (0: inactivo, 1: activo)
	 * @return int <0 si error, >0 si ok
	 */
	public function setStatus($id, $status)
	{
		$sql = "UPDATE " . MAIN_DB_PREFIX . "cfdixml_addenda";
		$sql .= " SET fk_status = " . (int) $status;
		$sql .= " WHERE rowid = " . (int) $id;

		$result = $this->db->query($sql);
		if ($result) {
			return 1;
		}
		$this->error = $this->db->lasterror();
		return -1;
	}

	/**
	 * Obtiene una addenda específica
	 *
	 * @param int $id ID de la addenda
	 * @return object|boolean Objeto con los datos o false si hay error
	 */
	public function fetch($id)
	{
		$sql = "SELECT rowid, addenda, class, fk_status";
		$sql .= " FROM " . MAIN_DB_PREFIX . "cfdixml_addenda";
		$sql .= " WHERE rowid = " . (int) $id;

		$resql = $this->db->query($sql);
		if ($resql) {
			if ($this->db->num_rows($resql)) {
				$obj = $this->db->fetch_object($resql);
				return $obj;
			}
			$this->db->free($resql);
			return false;
		}
		$this->error = $this->db->lasterror();
		return false;
	}

	/**
	 * Obtiene todas las addendas según su estado
	 *
	 * @param int $status Estado de las addendas (0: inactivas, 1: activas, null: todas)
	 * @return array|boolean Array de addendas o false si hay error
	 */
	public function fetchAll($status = null)
	{
		$addendas = array();

		$sql = "SELECT rowid, addenda, class, fk_status";
		$sql .= " FROM " . MAIN_DB_PREFIX . "cfdixml_addenda";
		if ($status !== null) {
			$sql .= " WHERE fk_status = " . (int) $status;
		}
		$sql .= " ORDER BY addenda ASC";

		$resql = $this->db->query($sql);
		if ($resql) {
			while ($obj = $this->db->fetch_object($resql)) {
				$addendas[] = $obj;
			}
			$this->db->free($resql);
			return $addendas;
		}
		$this->error = $this->db->lasterror();
		return false;
	}

	/**
	 * Comprueba si existe una addenda
	 *
	 * @param string $addendaName Nombre de la addenda
	 * @return boolean true si existe, false si no
	 */
	public function exists($addendaName)
	{
		$sql = "SELECT COUNT(*) as total FROM " . MAIN_DB_PREFIX . "cfdixml_addenda";
		$sql .= " WHERE addenda = '" . $this->db->escape($addendaName) . "'";

		$resql = $this->db->query($sql);
		if ($resql) {
			$obj = $this->db->fetch_object($resql);
			return ($obj->total > 0);
		}
		return false;
	}

	/**
	 * Obtiene el número total de addendas
	 *
	 * @param int $status Estado de las addendas (opcional)
	 * @return int|boolean Número de addendas o false si hay error
	 */
	public function countAddendas($status = null)
	{
		$sql = "SELECT COUNT(*) as total FROM " . MAIN_DB_PREFIX . "cfdixml_addenda";
		if ($status !== null) {
			$sql .= " WHERE fk_status = " . (int) $status;
		}

		$resql = $this->db->query($sql);
		if ($resql) {
			$obj = $this->db->fetch_object($resql);
			return (int) $obj->total;
		}
		return false;
	}

	/**
	 * Obtiene las addendas activas que están disponibles para un objeto específico
	 *
	 * @param string $objectType Tipo de objeto (facture, commande, etc.)
	 * @return array|boolean Array de addendas o false si hay error
	 */
	public function getActiveAddendas($objectType = '')
	{
		$sql = "SELECT rowid, addenda, class";
		$sql .= " FROM " . MAIN_DB_PREFIX . "cfdixml_addenda";
		$sql .= " WHERE fk_status = 1";
		if (!empty($objectType)) {
			$sql .= " AND class LIKE '%" . $this->db->escape($objectType) . "%'";
		}
		$sql .= " ORDER BY addenda ASC";

		$addendas = array();
		$resql = $this->db->query($sql);
		if ($resql) {
			while ($obj = $this->db->fetch_object($resql)) {
				$addendas[] = $obj;
			}
			return $addendas;
		}
		return false;
	}

	/**
	 * Crea los extrafields configurados para una addenda
	 *
	 * @param int $addendaId ID de la addenda
	 * @param ExtraFields $extrafields Objeto extrafields de Dolibarr
	 * @return int >0 si OK, <0 si error
	 */
	public function createConfiguredExtrafields($addendaId, $extrafields)
	{
		if (empty($addendaId) || !is_object($extrafields)) {
			$this->error = 'Invalid parameters';
			return -1;
		}

		// Obtener los extrafields configurados para esta addenda
		$sql = "SELECT *";
		$sql .= " FROM " . MAIN_DB_PREFIX . "cfdixml_addenda_extrafields";
		$sql .= " WHERE fk_addenda = " . (int) $addendaId;
		$sql .= " ORDER BY pos ASC";

		$resql = $this->db->query($sql);
		if (!$resql) {
			$this->error = $this->db->lasterror();
			return -1;
		}

		$num = $this->db->num_rows($resql);
		if ($num <= 0) {
			return 0; // No hay extrafields configurados
		}

		$error = 0;
		while ($obj = $this->db->fetch_object($resql)) {
			// Procesar los parámetros serializados
			$params = array();
			if (!empty($obj->param)) {
				$params = unserialize($obj->param);
				if ($params === false) {
					$params = array();
				}
			}

			// Procesar los parámetros adicionales
			$moreparams = array();
			if (!empty($obj->moreparams)) {
				$moreparams = unserialize($obj->moreparams);
				if ($moreparams === false) {
					$moreparams = array();
				}
			}

			// Verificar si el extrafield ya existe
			$result = $extrafields->fetch_name_optionals_label($obj->elementtype);
			if ($result < 0) {
				$error++;
				break;
			}

			// Si el extrafield no existe, crearlo
			if (!isset($extrafields->attributes[$obj->elementtype]['label'][$obj->attrname])) {
				$result = $extrafields->addExtraField(
					$obj->attrname,              // Código del atributo
					$obj->label,                 // Etiqueta
					$obj->type,                  // Tipo
					$obj->pos,                   // Posición
					$obj->size,                  // Tamaño
					$obj->elementtype,           // Tipo de elemento
					$obj->unique_field,          // Es único
					$obj->required,              // Es requerido
					$obj->default_value,         // Valor por defecto
					$params,                     // Parámetros
					$obj->alwayseditable,        // Siempre editable
					$obj->perms,                 // Permisos
					$obj->list,                  // Visibilidad
					$obj->help,                  // Ayuda
					$obj->computed,              // Valor calculado
					$obj->entity,                // Entidad
					$obj->langfile,              // Archivo de idioma
					$obj->enabled,               // Condición de habilitación
					$obj->totalizable,           // Es totalizable
					$obj->printable,             // Se muestra en PDF
					$moreparams                  // Parámetros adicionales
				);

				if ($result <= 0) {
					$this->error = 'Error creating extrafield ' . $obj->attrname;
					$error++;
					break;
				}
			}
		}

		if ($error) {
			return -1;
		}
		return 1;
	}

	/**
	 * Obtiene la lista de extrafields configurados para una addenda
	 *
	 * @param int $addendaId ID de la addenda
	 * @return array|boolean Array con la configuración de extrafields o false si hay error
	 */
	public function getConfiguredExtrafields($addendaId)
	{
		$sql = "SELECT *";
		$sql .= " FROM " . MAIN_DB_PREFIX . "cfdixml_addenda_extrafields";
		$sql .= " WHERE fk_addenda = " . (int) $addendaId;
		$sql .= " ORDER BY pos ASC";

		$extrafields = array();
		$resql = $this->db->query($sql);
		if ($resql) {
			while ($obj = $this->db->fetch_object($resql)) {
				// Procesar los parámetros serializados
				if (!empty($obj->param)) {
					$obj->param = unserialize($obj->param);
				}
				if (!empty($obj->moreparams)) {
					$obj->moreparams = unserialize($obj->moreparams);
				}
				$extrafields[] = $obj;
			}
			return $extrafields;
		}
		$this->error = $this->db->lasterror();
		return false;
	}
}
