<?php

/**
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

/**
 * \file       cfdixml/class/addenda/addendasoriana.class.php
 * \ingroup    cfdixml
 * \brief      Class file for Soriana Addenda
 */

require_once DOL_DOCUMENT_ROOT . '/core/class/commonobject.class.php';

class AddendaSoriana extends CommonObject
{
	public $element = 'addenda_soriana';
	public $table_element = 'cfdixml_addenda_soriana';

	public $fields = array(
		'rowid'           => array('type' => 'integer', 'label' => 'ID', 'enabled' => 1),
		'fk_societe'      => array('type' => 'integer', 'label' => 'ThirdParty', 'enabled' => 1),
		'codigo_proveedor' => array('type' => 'varchar(50)', 'label' => 'ProviderCode', 'enabled' => 1),
		'tipo_moneda'     => array('type' => 'integer', 'label' => 'CurrencyType', 'enabled' => 1),
		'tipo_bulto'      => array('type' => 'integer', 'label' => 'PackageType', 'enabled' => 1),
		'entity'          => array('type' => 'integer', 'label' => 'Entity', 'enabled' => 1, 'default' => 1)
	);

	public $rowid;
	public $fk_societe;
	public $codigo_proveedor;
	public $tipo_moneda;
	public $tipo_bulto;
	public $entity;

	public function __construct($db)
	{
		$this->db = $db;
		return 1;
	}

	/**
	 * Create object into database
	 *
	 * @param  User $user      User that creates
	 * @param  bool $notrigger false=launch triggers after, true=disable triggers
	 * @return int             <0 if KO, Id of created object if OK
	 */
	public function create(User $user, $notrigger = false)
	{
		return $this->createCommon($user, $notrigger);
	}

	/**
	 * Load object in memory from database
	 *
	 * @param int    $id   Id object
	 * @return int         <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetch($id)
	{
		return $this->fetchCommon($id);
	}

	/**
	 * Load object by third party
	 *
	 * @param int    $socid   Third party id
	 * @return int            <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetchBySociety($socid)
	{
		$sql = "SELECT t.rowid FROM " . $this->db->prefix() . $this->table_element . " as t";
		$sql .= " WHERE t.fk_societe = " . (int) $socid;
		$sql .= " AND t.entity IN (" . getEntity('addenda_soriana') . ")";

		$resql = $this->db->query($sql);
		if ($resql) {
			if ($obj = $this->db->fetch_object($resql)) {
				return $this->fetch($obj->rowid);
			}
			return 0;
		}
		return -1;
	}

	/**
	 * Update object into database
	 *
	 * @param  User $user      User that modifies
	 * @param  bool $notrigger false=launch triggers after, true=disable triggers
	 * @return int             <0 if KO, >0 if OK
	 */
	public function update(User $user, $notrigger = false)
	{
		return $this->updateCommon($user, $notrigger);
	}

	/**
	 * Delete object in database
	 *
	 * @param  User $user      User that deletes
	 * @param  bool $notrigger false=launch triggers after, true=disable triggers
	 * @return int             <0 if KO, >0 if OK
	 */
	public function delete(User $user, $notrigger = false)
	{
		return $this->deleteCommon($user, $notrigger);
	}
}

/**
 * Class para Addenda Soriana en Pedidos
 */
class AddendaSorianaCommande extends CommonObject
{
	public $element = 'addenda_soriana_commande';
	public $table_element = 'cfdixml_addenda_soriana_commande';

	public $fields = array(
		'rowid'               => array('type' => 'integer', 'label' => 'ID', 'enabled' => 1),
		'fk_commande'         => array('type' => 'integer', 'label' => 'Order', 'enabled' => 1),
		'remision'            => array('type' => 'varchar(50)', 'label' => 'Remission', 'enabled' => 1),
		'folio_pedido'        => array('type' => 'varchar(50)', 'label' => 'OrderRef', 'enabled' => 1),
		'cantidad_articulos'  => array('type' => 'integer', 'label' => 'ItemQuantity', 'enabled' => 1),
		'entity'              => array('type' => 'integer', 'label' => 'Entity', 'enabled' => 1),
		'fk_user_author'      => array('type' => 'integer', 'label' => 'UserAuthor', 'enabled' => 1),
		'fk_user_modif'       => array('type' => 'integer', 'label' => 'UserModif', 'enabled' => 1),
		'datec'               => array('type' => 'datetime', 'label' => 'DateCreation', 'enabled' => 1)
	);

	public $rowid;
	public $fk_commande;
	public $remision;
	public $folio_pedido;
	public $cantidad_articulos;
	public $entity;
	public $fk_user_author;
	public $fk_user_modif;
	public $datec;

	public function __construct($db)
	{
		$this->db = $db;
		return 1;
	}

	/**
	 * Create object into database
	 *
	 * @param  User $user      User that creates
	 * @param  bool $notrigger false=launch triggers after, true=disable triggers
	 * @return int             <0 if KO, Id of created object if OK
	 */
	public function create(User $user, $notrigger = false)
	{
		$this->fk_user_author = $user->id;
		$this->datec = dol_now();
		return $this->createCommon($user, $notrigger);
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param  int      $id       Id object
	 * @return int                <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetch($id)
	{
		return $this->fetchCommon($id);
	}

	/**
	 * Load object by order
	 *
	 * @param  int      $orderid    Order id
	 * @return int                  <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetchByOrder($orderid)
	{
		$sql = "SELECT t.rowid FROM " . $this->db->prefix() . $this->table_element . " as t";
		$sql .= " WHERE t.fk_commande = " . (int) $orderid;
		$sql .= " AND t.entity IN (" . getEntity($this->element) . ")";

		$resql = $this->db->query($sql);
		if ($resql) {
			if ($obj = $this->db->fetch_object($resql)) {
				return $this->fetch($obj->rowid);
			}
			return 0;
		}
		return -1;
	}

	/**
	 * Update object into database
	 *
	 * @param  User    $user      User that modifies
	 * @param  bool    $notrigger false=launch triggers after, true=disable triggers
	 * @return int                <0 if KO, >0 if OK
	 */
	public function update(User $user, $notrigger = false)
	{
		$this->fk_user_modif = $user->id;
		return $this->updateCommon($user, $notrigger);
	}

	/**
	 * Delete object in database
	 *
	 * @param  User    $user      User that deletes
	 * @param  bool    $notrigger false=launch triggers after, true=disable triggers
	 * @return int                <0 if KO, >0 if OK
	 */
	public function delete(User $user, $notrigger = false)
	{
		return $this->deleteCommon($user, $notrigger);
	}
}

/**
 * Class para Addenda Soriana en Envíos
 */
class AddendaSorianaExpedition extends CommonObject
{
	public $element = 'addenda_soriana_expedition';
	public $table_element = 'cfdixml_addenda_soriana_expedition';

	public $fields = array(
		'rowid'               => array('type' => 'integer', 'label' => 'ID', 'enabled' => 1),
		'fk_expedition'       => array('type' => 'integer', 'label' => 'Shipping', 'enabled' => 1),
		'remision'            => array('type' => 'varchar(50)', 'label' => 'Remission', 'enabled' => 1),
		'consecutivo'         => array('type' => 'integer', 'label' => 'Consecutive', 'enabled' => 1),
		'fecha_remision'      => array('type' => 'datetime', 'label' => 'RemissionDate', 'enabled' => 1),
		'tipo_bulto'          => array('type' => 'integer', 'label' => 'PackageType', 'enabled' => 1),
		'cantidad_bultos'     => array('type' => 'integer', 'label' => 'PackageQuantity', 'enabled' => 1),
		'fecha_entrega'       => array('type' => 'datetime', 'label' => 'DeliveryDate', 'enabled' => 1),
		'empaque_cajas'       => array('type' => 'boolean', 'label' => 'BoxPacking', 'enabled' => 1),
		'empaque_tarimas'     => array('type' => 'boolean', 'label' => 'PalletPacking', 'enabled' => 1),
		'cantidad_cajas_tarimas' => array('type' => 'integer', 'label' => 'BoxPalletQuantity', 'enabled' => 1),
		'cita'                => array('type' => 'varchar(50)', 'label' => 'Appointment', 'enabled' => 1),
		'entity'              => array('type' => 'integer', 'label' => 'Entity', 'enabled' => 1),
		'fk_user_author'      => array('type' => 'integer', 'label' => 'UserAuthor', 'enabled' => 1),
		'fk_user_modif'       => array('type' => 'integer', 'label' => 'UserModif', 'enabled' => 1),
		'datec'               => array('type' => 'datetime', 'label' => 'DateCreation', 'enabled' => 1)
	);

	public $rowid;
	public $fk_expedition;
	public $remision;
	public $consecutivo;
	public $fecha_remision;
	public $tipo_bulto;
	public $cantidad_bultos;
	public $fecha_entrega;
	public $empaque_cajas;
	public $empaque_tarimas;
	public $cantidad_cajas_tarimas;
	public $cita;
	public $entity;
	public $fk_user_author;
	public $fk_user_modif;
	public $datec;

	public function __construct($db)
	{
		$this->db = $db;
		return 1;
	}
	/**
	 * Create object into database
	 *
	 * @param  User $user      User that creates
	 * @param  bool $notrigger false=launch triggers after, true=disable triggers
	 * @return int             <0 if KO, Id of created object if OK
	 */
	public function create(User $user, $notrigger = false)
	{
		$this->fk_user_author = $user->id;
		$this->datec = dol_now();
		return $this->createCommon($user, $notrigger);
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param  int      $id       Id object
	 * @return int                <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetch($id)
	{
		return $this->fetchCommon($id);
	}

	/**
	 * Load object by expedition
	 *
	 * @param  int      $expeditionid    Expedition id
	 * @return int                       <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetchByExpedition($expeditionid)
	{
		$sql = "SELECT t.rowid FROM " . $this->db->prefix() . $this->table_element . " as t";
		$sql .= " WHERE t.fk_expedition = " . (int) $expeditionid;
		$sql .= " AND t.entity IN (" . getEntity($this->element) . ")";

		$resql = $this->db->query($sql);
		if ($resql) {
			if ($obj = $this->db->fetch_object($resql)) {
				return $this->fetch($obj->rowid);
			}
			return 0;
		}
		return -1;
	}

	/**
	 * Update object into database
	 *
	 * @param  User    $user      User that modifies
	 * @param  bool    $notrigger false=launch triggers after, true=disable triggers
	 * @return int                <0 if KO, >0 if OK
	 */
	public function update(User $user, $notrigger = false)
	{
		$this->fk_user_modif = $user->id;
		return $this->updateCommon($user, $notrigger);
	}

	/**
	 * Delete object in database
	 *
	 * @param  User    $user      User that deletes
	 * @param  bool    $notrigger false=launch triggers after, true=disable triggers
	 * @return int                <0 if KO, >0 if OK
	 */
	public function delete(User $user, $notrigger = false)
	{
		return $this->deleteCommon($user, $notrigger);
	}

	/**
	 * Load lines of expedition addenda
	 *
	 * @return AddendaSorianaExpeditionDet[]    Array of expedition detail lines
	 */
	public function lines()
	{
		$lines = array();

		$sql = "SELECT rowid FROM " . $this->db->prefix() . "cfdixml_addenda_soriana_expedition_det";
		$sql .= " WHERE fk_expedition = " . (int) $this->fk_expedition;
		$sql .= " AND entity = " . $this->entity;
		$sql .= " ORDER BY rowid";

		$resql = $this->db->query($sql);
		if ($resql) {
			require_once DOL_DOCUMENT_ROOT . '/expedition/class/expedition.class.php';
			$lineObj = new AddendaSorianaExpeditionDet($this->db);

			while ($obj = $this->db->fetch_object($resql)) {
				$result = $lineObj->fetch($obj->rowid);
				if ($result > 0) {
					$lines[] = $lineObj;
				}
			}
			$this->db->free($resql);
			return $lines;
		} else {
			$this->error = $this->db->lasterror();
			return -1;
		}
	}
}

/**
 * Class para detalles de artículos en envíos
 */
class AddendaSorianaExpeditionDet extends CommonObject
{
	public $element = 'addenda_soriana_expedition_det';
	public $table_element = 'cfdixml_addenda_soriana_expedition_det';

	public $fields = array(
		'rowid'              => array('type' => 'integer', 'label' => 'ID', 'enabled' => 1),
		'fk_expedition_line' => array('type' => 'integer', 'label' => 'ShippingLine', 'enabled' => 1),
		'fk_expedition'      => array('type' => 'integer', 'label' => 'Shipping', 'enabled' => 1),
		'codigo'             => array('type' => 'varchar(50)', 'label' => 'Code', 'enabled' => 1),
		'cantidad_unidad'    => array('type' => 'double(24,8)', 'label' => 'Quantity', 'enabled' => 1),
		'costo_neto'         => array('type' => 'double(24,8)', 'label' => 'NetCost', 'enabled' => 1),
		'porcentaje_ieps'    => array('type' => 'double(24,8)', 'label' => 'IEPSRate', 'enabled' => 1),
		'porcentaje_iva'     => array('type' => 'double(24,8)', 'label' => 'VATRate', 'enabled' => 1),
		'entity'             => array('type' => 'integer', 'label' => 'Entity', 'enabled' => 1)
	);

	public $rowid;
	public $fk_expedition_line;
	public $fk_expedition;
	public $codigo;
	public $cantidad_unidad;
	public $costo_neto;
	public $porcentaje_ieps;
	public $porcentaje_iva;
	public $entity;

	public function __construct($db)
	{
		$this->db = $db;
		return 1;
	}

	public function create(User $user, $notrigger = false)
	{
		return $this->createCommon($user, $notrigger);
	}

	public function fetch($id)
	{
		return $this->fetchCommon($id);
	}

	public function fetchByExpeditionLine($expedition_line_id)
	{
		$sql = "SELECT t.rowid FROM " . $this->db->prefix() . $this->table_element . " as t";
		$sql .= " WHERE t.fk_expedition_line = " . (int) $expedition_line_id;
		$sql .= " AND t.entity IN (" . getEntity('addenda_soriana_expedition_det') . ")";

		$resql = $this->db->query($sql);
		if ($resql) {
			if ($obj = $this->db->fetch_object($resql)) {
				return $this->fetch($obj->rowid);
			}
			return 0;
		}
		return -1;
	}

	public function update(User $user, $notrigger = false)
	{
		return $this->updateCommon($user, $notrigger);
	}

	public function delete(User $user, $notrigger = false)
	{
		return $this->deleteCommon($user, $notrigger);
	}
}
