<?php

class SQLiteController
{
	private $dbPath;
	private $pdo;

	/**
	 * Constructor
	 * @param string $dbPath Ruta al archivo SQLite
	 */
	public function __construct(string $dbPath)
	{
		$this->dbPath = $dbPath;

		// Verificar si el driver SQLite está disponible
		if (!extension_loaded('pdo_sqlite')) {
			throw new RuntimeException(
				"El driver SQLite no está instalado. Instale el paquete `pdo_sqlite` para continuar."
			);
		}
	}

	/**
	 * Conecta a la base de datos SQLite
	 * @throws RuntimeException Si no puede conectarse
	 */
	public function connect()
	{
		if (!file_exists($this->dbPath)) {
			throw new RuntimeException("El archivo SQLite no existe: {$this->dbPath}");
		}

		try {
			$this->pdo = new PDO("sqlite:" . $this->dbPath);
			$this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
		} catch (PDOException $e) {
			throw new RuntimeException("No se pudo conectar a la base de datos SQLite: " . $e->getMessage());
		}
	}

	/**
	 * Ejecuta una consulta y devuelve los resultados
	 * @param string $query Consulta SQL
	 * @param array $params Parámetros de la consulta
	 * @return array Resultados de la consulta
	 * @throws RuntimeException Si la consulta falla
	 */
	public function executeQuery(string $query, array $params = []): array
	{
		if (!$this->pdo) {
			throw new RuntimeException("No se ha establecido conexión con la base de datos.");
		}

		try {
			$stmt = $this->pdo->prepare($query);
			$stmt->execute($params);
			return $stmt->fetchAll(PDO::FETCH_ASSOC);
		} catch (PDOException $e) {
			throw new RuntimeException("Error al ejecutar la consulta: " . $e->getMessage());
		}
	}

	/**
	 * Cierra la conexión con la base de datos
	 */
	public function closeConnection()
	{
		$this->pdo = null;
	}
}
